/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#include <sal/config.h>
#include <comphelper/solarmutex.hxx>
#include <drawinglayer/primitive2d/BufferedDecompositionFlusher.hxx>

namespace drawinglayer::primitive2d
{
/**
    This is a "garbage collection" approach to flushing.

    We store entries in a set. Every 2 seconds, we scan the set for entries that have not
    been used for 10 seconds or more, and if so, we flush the buffer primitives in those entries.

    This mechanism is __deliberately__ not perfect.
    Sometimes things will be flushed a little too soon, sometimes things will wait a little too long,
    since we only have a granularity of 2 seconds.
    But what is gains from not being perfect, is scalability.

    It is very simple, scales to lots and lots of primitives without needing lots of timers, and performs
    very little work in the common case.

    Shutdown notes
    --------------------
    The process of handling shutdown is more complicated here than it should be, because we are interacting with
    various vcl-level things (by virtue of calling into drawinglayer primitives that use vcl facilities), but we
    do not have access to vcl-level API here (like SolarMutexReleaser and vcl::Timer).
*/

static BufferedDecompositionFlusher* getInstance()
{
    static std::unique_ptr<BufferedDecompositionFlusher> gaTimer(new BufferedDecompositionFlusher);
    return gaTimer.get();
}

// static
void BufferedDecompositionFlusher::shutdown()
{
    BufferedDecompositionFlusher* pFlusher = getInstance();
    pFlusher->onTeardown();
    // We have to wait for the thread to exit, otherwise we might end up with the background thread
    // trying to process stuff while it has things ripped out underneath it.
    pFlusher->join();
}

// static
void BufferedDecompositionFlusher::update(const BufferedDecompositionPrimitive2D* p)
{
    getInstance()->updateImpl(p);
}

// static
void BufferedDecompositionFlusher::update(const BufferedDecompositionGroupPrimitive2D* p)
{
    getInstance()->updateImpl(p);
}

// static
void BufferedDecompositionFlusher::remove(const BufferedDecompositionPrimitive2D* p)
{
    getInstance()->removeImpl(p);
}

// static
void BufferedDecompositionFlusher::remove(const BufferedDecompositionGroupPrimitive2D* p)
{
    getInstance()->removeImpl(p);
}

BufferedDecompositionFlusher::BufferedDecompositionFlusher() { create(); }

void BufferedDecompositionFlusher::updateImpl(const BufferedDecompositionPrimitive2D* p)
{
    std::unique_lock l(maMutex);
    if (!mbShutdown)
    {
        unotools::WeakReference<BufferedDecompositionPrimitive2D> xRef(
            const_cast<BufferedDecompositionPrimitive2D*>(p));
        maRegistered1.insert({ p, std::move(xRef) });
    }
}

void BufferedDecompositionFlusher::updateImpl(const BufferedDecompositionGroupPrimitive2D* p)
{
    std::unique_lock l(maMutex);
    if (!mbShutdown)
    {
        unotools::WeakReference<BufferedDecompositionGroupPrimitive2D> xRef(
            const_cast<BufferedDecompositionGroupPrimitive2D*>(p));
        maRegistered2.insert({ p, std::move(xRef) });
    }
}

void BufferedDecompositionFlusher::removeImpl(const BufferedDecompositionPrimitive2D* p)
{
    std::unique_lock l(maMutex);
    if (!mbShutdown)
        maRegistered1.erase(p);
}

void BufferedDecompositionFlusher::removeImpl(const BufferedDecompositionGroupPrimitive2D* p)
{
    std::unique_lock l(maMutex);
    if (!mbShutdown)
        maRegistered2.erase(p);
}

void SAL_CALL BufferedDecompositionFlusher::run()
{
    setName("BufferedDecompositionFlusher");
    for (;;)
    {
        auto aNow = std::chrono::steady_clock::now();
        std::vector<rtl::Reference<BufferedDecompositionPrimitive2D>> aRemoved1;
        std::vector<rtl::Reference<BufferedDecompositionGroupPrimitive2D>> aRemoved2;
        {
            std::unique_lock l1(maMutex);
            // exit if we have been shutdown
            if (mbShutdown)
                break;
            for (auto it = maRegistered1.begin(); it != maRegistered1.end();)
            {
                rtl::Reference<BufferedDecompositionPrimitive2D> xPrimitive = it->second.get();
                if (!xPrimitive)
                    it = maRegistered1.erase(it);
                else if (aNow - xPrimitive->maLastAccess.load() > std::chrono::seconds(10))
                {
                    aRemoved1.push_back(std::move(xPrimitive));
                    it = maRegistered1.erase(it);
                }
                else
                    ++it;
            }
            for (auto it = maRegistered2.begin(); it != maRegistered2.end();)
            {
                rtl::Reference<BufferedDecompositionGroupPrimitive2D> xPrimitive = it->second.get();
                if (!xPrimitive)
                    it = maRegistered2.erase(it);
                else if (aNow - xPrimitive->maLastAccess.load() > std::chrono::seconds(10))
                {
                    aRemoved2.push_back(std::move(xPrimitive));
                    it = maRegistered2.erase(it);
                }
                else
                    ++it;
            }
        }

        {
            // some parts of skia do not take kindly to being accessed from multiple threads
            osl::Guard<comphelper::SolarMutex> aGuard(comphelper::SolarMutex::get());

            for (const auto& xPrim : aRemoved1)
            {
                xPrim->setBuffered2DDecomposition(nullptr);
            }
            for (const auto& xPrim : aRemoved2)
            {
                xPrim->setBuffered2DDecomposition(Primitive2DContainer{});
            }

            // Clear these while under the SolarMutex, just in case we are the sole surviving reference,
            // and we might trigger destruction of related vcl resources.
            aRemoved1.clear();
            aRemoved2.clear();
        }

        {
            std::unique_lock l(maMutex);
            maDelayOrTerminate.wait_for(l, std::chrono::seconds(2), [this] { return mbShutdown; });
        }
    }
}

/// Only called by FlusherDeinit
void BufferedDecompositionFlusher::onTeardown()
{
    {
        std::unique_lock l2(maMutex);
        mbShutdown = true;
        maRegistered1.clear();
        maRegistered2.clear();
    }
    maDelayOrTerminate.notify_all();
}

} // end of namespace drawinglayer::primitive2d

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
