/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2023      Yi-Yen Chung <eric681@andestech.com> (Copyright owned by Andes Technology)
 *   2024      Zhijin Zeng <zhijin.zeng@spacemit.com>
 *   2025      Russell Graves <rgraves@sevarg.net>
 */

#define SIMDE_TEST_ARM_NEON_INSN qrshl

#include "test-neon.h"

/* Check that both of these work */
#if defined(__cplusplus)
  #include <simde/arm/neon/qrshl.h>
#else
  #include <simde/arm/neon.h>
#endif

#if 0
#define PROBABILITY 80
#define probability(p) (rand() < (((int64_t) RAND_MAX * (p)) / 100))
#endif

static int
test_simde_vqrshlb_s8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    int8_t a;
    int8_t b;
    int8_t r;
  } test_vec[] = {
    { -INT8_C(  41),
      -INT8_C(  84),
       INT8_C(   0) },
    {  INT8_C(  91),
       INT8_C(  87),
           INT8_MAX },
    { -INT8_C(  52),
      -INT8_C( 119),
       INT8_C(   0) },
    { -INT8_C(  86),
       INT8_C( 101),
           INT8_MIN },
    {  INT8_C(  35),
      -INT8_C(  43),
       INT8_C(   0) },
    { -INT8_C(  13),
       INT8_C( 121),
           INT8_MIN },
    { -INT8_C(  46),
       INT8_C(  91),
           INT8_MIN },
    { -INT8_C(  17),
      -INT8_C(  79),
       INT8_C(   0) },
    {  INT8_C(  72),
      -INT8_C(  10),
       INT8_C(   0) },
    { -INT8_C(  93),
      -INT8_C(   9),
       INT8_C(   0) },
    {  INT8_C(   8),
      -INT8_C(   9),
       INT8_C(   0) },
    { -INT8_C(  17),
      -INT8_C(   9),
       INT8_C(   0) },
    {  INT8_C(  21),
      -INT8_C(   9),
       INT8_C(   0) },
    { -INT8_C(  92),
      -INT8_C(   8),
       INT8_C(   0) },
    { -INT8_C(  37),
      -INT8_C(   8),
       INT8_C(   0) },
    {  INT8_C(  54),
      -INT8_C(   8),
       INT8_C(   0) },
    {  INT8_C(  12),
      -INT8_C(   8),
       INT8_C(   0) },
    { -INT8_C( 103),
      -INT8_C(   7),
      -INT8_C(   1) },
    {  INT8_C(  52),
      -INT8_C(   7),
       INT8_C(   0) },
    {  INT8_C( 103),
      -INT8_C(   7),
       INT8_C(   1) },
    { -INT8_C( 109),
      -INT8_C(   7),
      -INT8_C(   1) },
    { -INT8_C(  99),
      -INT8_C(   6),
      -INT8_C(   2) },
    {  INT8_C(  27),
      -INT8_C(   6),
       INT8_C(   0) },
    {  INT8_C( 107),
      -INT8_C(   6),
       INT8_C(   2) },
    {  INT8_C(  73),
      -INT8_C(   6),
       INT8_C(   1) },
    {  INT8_C( 118),
      -INT8_C(   5),
       INT8_C(   4) },
    { -INT8_C(  62),
      -INT8_C(   5),
      -INT8_C(   2) },
    {  INT8_C(  21),
      -INT8_C(   5),
       INT8_C(   1) },
    {  INT8_C(   0),
      -INT8_C(   5),
       INT8_C(   0) },
    {  INT8_C( 108),
      -INT8_C(   4),
       INT8_C(   7) },
    {  INT8_C( 123),
      -INT8_C(   4),
       INT8_C(   8) },
    {  INT8_C(  35),
      -INT8_C(   4),
       INT8_C(   2) },
    {  INT8_C(  65),
      -INT8_C(   4),
       INT8_C(   4) },
    {  INT8_C( 110),
      -INT8_C(   3),
       INT8_C(  14) },
    { -INT8_C( 100),
      -INT8_C(   3),
      -INT8_C(  12) },
    {  INT8_C(  19),
      -INT8_C(   3),
       INT8_C(   2) },
    { -INT8_C(  55),
      -INT8_C(   3),
      -INT8_C(   7) },
    { -INT8_C( 116),
      -INT8_C(   2),
      -INT8_C(  29) },
    { -INT8_C(  59),
      -INT8_C(   2),
      -INT8_C(  15) },
    {  INT8_C(  17),
      -INT8_C(   2),
       INT8_C(   4) },
    {  INT8_C(  47),
      -INT8_C(   2),
       INT8_C(  12) },
    { -INT8_C(  51),
      -INT8_C(   1),
      -INT8_C(  25) },
    {  INT8_C(   0),
      -INT8_C(   1),
       INT8_C(   0) },
    {  INT8_C(  68),
      -INT8_C(   1),
       INT8_C(  34) },
    {  INT8_C( 113),
      -INT8_C(   1),
       INT8_C(  57) },
    { -INT8_C(  37),
       INT8_C(   0),
      -INT8_C(  37) },
    {  INT8_C( 122),
       INT8_C(   0),
       INT8_C( 122) },
    {  INT8_C( 125),
       INT8_C(   0),
       INT8_C( 125) },
    {  INT8_C( 116),
       INT8_C(   0),
       INT8_C( 116) },
    { -INT8_C(  82),
       INT8_C(   0),
      -INT8_C(  82) },
    { -INT8_C(  28),
       INT8_C(   0),
      -INT8_C(  28) },
    {  INT8_C(   7),
       INT8_C(   0),
       INT8_C(   7) },
    {  INT8_C(  76),
       INT8_C(   1),
           INT8_MAX },
    {  INT8_C(   0),
       INT8_C(   1),
       INT8_C(   0) },
    {  INT8_C( 114),
       INT8_C(   1),
           INT8_MAX },
    { -INT8_C( 107),
       INT8_C(   1),
           INT8_MIN },
    {  INT8_C( 118),
       INT8_C(   2),
           INT8_MAX },
    {  INT8_C(  52),
       INT8_C(   2),
           INT8_MAX },
    { -INT8_C(  85),
       INT8_C(   2),
           INT8_MIN },
    {  INT8_C( 118),
       INT8_C(   2),
           INT8_MAX },
    { -INT8_C(  95),
       INT8_C(   3),
           INT8_MIN },
    {  INT8_C(  38),
       INT8_C(   3),
           INT8_MAX },
    { -INT8_C( 103),
       INT8_C(   3),
           INT8_MIN },
    { -INT8_C(  30),
       INT8_C(   3),
           INT8_MIN },
    { -INT8_C( 108),
       INT8_C(   4),
           INT8_MIN },
    {  INT8_C(  54),
       INT8_C(   4),
           INT8_MAX },
    { -INT8_C(  10),
       INT8_C(   4),
           INT8_MIN },
    {  INT8_C(  93),
       INT8_C(   4),
           INT8_MAX },
    { -INT8_C(  62),
       INT8_C(   5),
           INT8_MIN },
    { -INT8_C(  69),
       INT8_C(   5),
           INT8_MIN },
    {  INT8_C( 110),
       INT8_C(   5),
           INT8_MAX },
    { -INT8_C(  15),
       INT8_C(   5),
           INT8_MIN },
    { -INT8_C( 120),
       INT8_C(   6),
           INT8_MIN },
    {  INT8_C( 110),
       INT8_C(   6),
           INT8_MAX },
    {  INT8_C(  53),
       INT8_C(   6),
           INT8_MAX },
    { -INT8_C(   7),
       INT8_C(   6),
           INT8_MIN },
    {  INT8_C(  73),
       INT8_C(   7),
           INT8_MAX },
    { -INT8_C(  81),
       INT8_C(   7),
           INT8_MIN },
    {  INT8_C( 119),
       INT8_C(   7),
           INT8_MAX },
    { -INT8_C(  67),
       INT8_C(   7),
           INT8_MIN },
    {  INT8_C(  93),
       INT8_C(   8),
           INT8_MAX },
    {  INT8_C(  91),
       INT8_C(   8),
           INT8_MAX },
    { -INT8_C(  60),
       INT8_C(   8),
           INT8_MIN },
    { -INT8_C(  87),
       INT8_C(   8),
           INT8_MIN },
    {  INT8_C(  91),
       INT8_C(   9),
           INT8_MAX },
    {  INT8_C(  55),
       INT8_C(   9),
           INT8_MAX },
    {  INT8_C(  63),
       INT8_C(   9),
           INT8_MAX },
    { -INT8_C(  46),
       INT8_C(   9),
           INT8_MIN },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    int8_t r = simde_vqrshlb_s8(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_i8(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    int8_t a = simde_test_codegen_random_i8();
    int8_t b = simde_test_codegen_random_i8();
    int8_t r = simde_vqrshlb_s8(a, b);

    simde_test_codegen_write_i8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -40 ; i < 40 ; i++) {
    int8_t a = simde_test_codegen_random_i8();
    int8_t b = i / 4;
    int8_t r = simde_vqrshlb_s8(a, b);

    simde_test_codegen_write_i8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlh_s16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    int16_t a;
    int16_t b;
    int16_t r;
  } test_vec[] = {
    {  INT16_C( 28496),
       INT16_C( 15818),
       INT16_C(     0) },
    { -INT16_C( 10793),
      -INT16_C( 19992),
       INT16_C(     0) },
    {  INT16_C( 31373),
       INT16_C( 20741),
             INT16_MAX },
    { -INT16_C( 22758),
      -INT16_C( 25378),
       INT16_C(     0) },
    {  INT16_C( 29475),
      -INT16_C( 21986),
             INT16_MAX },
    { -INT16_C( 29503),
       INT16_C( 12223),
       INT16_C(     0) },
    {  INT16_C( 13721),
      -INT16_C(  6022),
             INT16_MAX },
    {  INT16_C(  3545),
       INT16_C( 10606),
             INT16_MAX },
    {  INT16_C( 14716),
      -INT16_C(    20),
       INT16_C(     0) },
    {  INT16_C( 21351),
      -INT16_C(    19),
       INT16_C(     0) },
    {  INT16_C( 20238),
      -INT16_C(    19),
       INT16_C(     0) },
    { -INT16_C( 25851),
      -INT16_C(    18),
       INT16_C(     0) },
    {  INT16_C(  2761),
      -INT16_C(    18),
       INT16_C(     0) },
    { -INT16_C(  7188),
      -INT16_C(    17),
       INT16_C(     0) },
    { -INT16_C( 13647),
      -INT16_C(    17),
       INT16_C(     0) },
    { -INT16_C( 11137),
      -INT16_C(    16),
       INT16_C(     0) },
    { -INT16_C( 25283),
      -INT16_C(    16),
       INT16_C(     0) },
    { -INT16_C(   385),
      -INT16_C(    15),
       INT16_C(     0) },
    {  INT16_C( 15913),
      -INT16_C(    15),
       INT16_C(     0) },
    { -INT16_C( 15571),
      -INT16_C(    14),
      -INT16_C(     1) },
    { -INT16_C( 22412),
      -INT16_C(    14),
      -INT16_C(     1) },
    {  INT16_C( 19883),
      -INT16_C(    13),
       INT16_C(     2) },
    {  INT16_C(  6581),
      -INT16_C(    13),
       INT16_C(     1) },
    {  INT16_C( 12663),
      -INT16_C(    12),
       INT16_C(     3) },
    { -INT16_C(  8622),
      -INT16_C(    12),
      -INT16_C(     2) },
    {  INT16_C( 24708),
      -INT16_C(    11),
       INT16_C(    12) },
    { -INT16_C( 30419),
      -INT16_C(    11),
      -INT16_C(    15) },
    { -INT16_C(  2309),
      -INT16_C(    10),
      -INT16_C(     2) },
    { -INT16_C(  6252),
      -INT16_C(    10),
      -INT16_C(     6) },
    {  INT16_C( 17881),
      -INT16_C(     9),
       INT16_C(    35) },
    {  INT16_C( 22962),
      -INT16_C(     9),
       INT16_C(    45) },
    { -INT16_C(  4326),
      -INT16_C(     8),
      -INT16_C(    17) },
    { -INT16_C( 26122),
      -INT16_C(     8),
      -INT16_C(   102) },
    {  INT16_C(  8430),
      -INT16_C(     7),
       INT16_C(    66) },
    {  INT16_C(  7127),
      -INT16_C(     7),
       INT16_C(    56) },
    {  INT16_C( 19427),
      -INT16_C(     6),
       INT16_C(   304) },
    { -INT16_C( 28989),
      -INT16_C(     6),
      -INT16_C(   453) },
    {  INT16_C( 30873),
      -INT16_C(     5),
       INT16_C(   965) },
    {  INT16_C(  4263),
      -INT16_C(     5),
       INT16_C(   133) },
    { -INT16_C(  1367),
      -INT16_C(     4),
      -INT16_C(    85) },
    {  INT16_C( 12014),
      -INT16_C(     4),
       INT16_C(   751) },
    {  INT16_C(  7002),
      -INT16_C(     3),
       INT16_C(   875) },
    {  INT16_C( 22199),
      -INT16_C(     3),
       INT16_C(  2775) },
    {  INT16_C( 19217),
      -INT16_C(     2),
       INT16_C(  4804) },
    { -INT16_C(  5571),
      -INT16_C(     2),
      -INT16_C(  1393) },
    { -INT16_C(  4207),
      -INT16_C(     1),
      -INT16_C(  2103) },
    { -INT16_C( 21693),
      -INT16_C(     1),
      -INT16_C( 10846) },
    {  INT16_C( 15071),
       INT16_C(     0),
       INT16_C( 15071) },
    { -INT16_C( 12988),
       INT16_C(     0),
      -INT16_C( 12988) },
    {  INT16_C(  7002),
       INT16_C(     0),
       INT16_C(  7002) },
    {  INT16_C( 15848),
       INT16_C(     1),
       INT16_C( 31696) },
    { -INT16_C( 21401),
       INT16_C(     1),
             INT16_MIN },
    {  INT16_C(   203),
       INT16_C(     2),
       INT16_C(   812) },
    {  INT16_C( 29220),
       INT16_C(     2),
             INT16_MAX },
    { -INT16_C( 12784),
       INT16_C(     3),
             INT16_MIN },
    { -INT16_C(   404),
       INT16_C(     3),
      -INT16_C(  3232) },
    { -INT16_C( 14340),
       INT16_C(     4),
             INT16_MIN },
    { -INT16_C( 19687),
       INT16_C(     4),
             INT16_MIN },
    {  INT16_C( 10781),
       INT16_C(     5),
             INT16_MAX },
    {  INT16_C( 23295),
       INT16_C(     5),
             INT16_MAX },
    { -INT16_C( 28652),
       INT16_C(     6),
             INT16_MIN },
    {  INT16_C( 22602),
       INT16_C(     6),
             INT16_MAX },
    {  INT16_C( 10555),
       INT16_C(     7),
             INT16_MAX },
    {  INT16_C( 32658),
       INT16_C(     7),
             INT16_MAX },
    { -INT16_C(  4874),
       INT16_C(     8),
             INT16_MIN },
    { -INT16_C(  8550),
       INT16_C(     8),
             INT16_MIN },
    {  INT16_C(   297),
       INT16_C(     9),
             INT16_MAX },
    { -INT16_C(  2934),
       INT16_C(     9),
             INT16_MIN },
    { -INT16_C( 20735),
       INT16_C(    10),
             INT16_MIN },
    {  INT16_C(  4454),
       INT16_C(    10),
             INT16_MAX },
    { -INT16_C( 11395),
       INT16_C(    11),
             INT16_MIN },
    {  INT16_C( 30991),
       INT16_C(    11),
             INT16_MAX },
    {  INT16_C( 10394),
       INT16_C(    12),
             INT16_MAX },
    { -INT16_C( 18644),
       INT16_C(    12),
             INT16_MIN },
    {  INT16_C( 11090),
       INT16_C(    13),
             INT16_MAX },
    {  INT16_C( 26385),
       INT16_C(    13),
             INT16_MAX },
    {  INT16_C( 23483),
       INT16_C(    14),
             INT16_MAX },
    { -INT16_C(  2369),
       INT16_C(    14),
             INT16_MIN },
    {  INT16_C( 20868),
       INT16_C(    15),
             INT16_MAX },
    {  INT16_C( 31349),
       INT16_C(    15),
             INT16_MAX },
    {  INT16_C(  4157),
       INT16_C(    16),
             INT16_MAX },
    {  INT16_C( 26201),
       INT16_C(    16),
             INT16_MAX },
    { -INT16_C(  7407),
       INT16_C(    17),
             INT16_MIN },
    {  INT16_C(  4954),
       INT16_C(    17),
             INT16_MAX },
    { -INT16_C( 16238),
       INT16_C(    18),
             INT16_MIN },
    {  INT16_C(  3876),
       INT16_C(    18),
             INT16_MAX },
    {  INT16_C( 13459),
       INT16_C(    19),
             INT16_MAX },
    {  INT16_C( 11656),
       INT16_C(    19),
             INT16_MAX },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    int16_t r = simde_vqrshlh_s16(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_i16(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    int16_t a = simde_test_codegen_random_i16();
    int16_t b = simde_test_codegen_random_i16();
    int16_t r = simde_vqrshlh_s16(a, b);

    simde_test_codegen_write_i16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -40 ; i < 40 ; i++) {
    int16_t a = simde_test_codegen_random_i16();
    int16_t b = i / 2;
    int16_t r = simde_vqrshlh_s16(a, b);

    simde_test_codegen_write_i16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshls_s32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    int32_t a;
    int32_t b;
    int32_t r;
  } test_vec[] = {
    { -INT32_C(   174426796),
       INT32_C(  1634665650),
       INT32_C(           0) },
    { -INT32_C(   475937015),
       INT32_C(   616240279),
       INT32_C(           0) },
    {  INT32_C(   519741645),
      -INT32_C(   557045947),
                   INT32_MAX },
    {  INT32_C(  1344220536),
      -INT32_C(   911287691),
                   INT32_MAX },
    {  INT32_C(    29247567),
       INT32_C(  1432497484),
                   INT32_MAX },
    { -INT32_C(  1942486028),
      -INT32_C(   441388265),
                   INT32_MIN },
    { -INT32_C(   654070892),
       INT32_C(  1253560274),
       INT32_C(           0) },
    {  INT32_C(  1922815741),
       INT32_C(     3885489),
       INT32_C(           0) },
    { -INT32_C(   570230382),
      -INT32_C(          40),
       INT32_C(           0) },
    {  INT32_C(   439641126),
      -INT32_C(          39),
       INT32_C(           0) },
    {  INT32_C(  2141613160),
      -INT32_C(          39),
       INT32_C(           0) },
    { -INT32_C(   194750624),
      -INT32_C(          38),
       INT32_C(           0) },
    { -INT32_C(   724735998),
      -INT32_C(          38),
       INT32_C(           0) },
    {  INT32_C(   874480695),
      -INT32_C(          37),
       INT32_C(           0) },
    {  INT32_C(   195476058),
      -INT32_C(          37),
       INT32_C(           0) },
    { -INT32_C(  1794318077),
      -INT32_C(          36),
       INT32_C(           0) },
    {  INT32_C(     7605978),
      -INT32_C(          36),
       INT32_C(           0) },
    { -INT32_C(   635721614),
      -INT32_C(          35),
       INT32_C(           0) },
    {  INT32_C(  1952104724),
      -INT32_C(          35),
       INT32_C(           0) },
    {  INT32_C(   443072024),
      -INT32_C(          34),
       INT32_C(           0) },
    {  INT32_C(  1592735014),
      -INT32_C(          34),
       INT32_C(           0) },
    {  INT32_C(   345116345),
      -INT32_C(          33),
       INT32_C(           0) },
    { -INT32_C(   887146040),
      -INT32_C(          33),
       INT32_C(           0) },
    { -INT32_C(   178181350),
      -INT32_C(          32),
       INT32_C(           0) },
    { -INT32_C(  1393175239),
      -INT32_C(          32),
       INT32_C(           0) },
    { -INT32_C(  1853484931),
      -INT32_C(          31),
      -INT32_C(           1) },
    { -INT32_C(   368647982),
      -INT32_C(          31),
       INT32_C(           0) },
    { -INT32_C(   989499745),
      -INT32_C(          30),
      -INT32_C(           1) },
    {  INT32_C(  1562637476),
      -INT32_C(          30),
       INT32_C(           1) },
    { -INT32_C(   898517502),
      -INT32_C(          29),
      -INT32_C(           2) },
    {  INT32_C(   160797167),
      -INT32_C(          29),
       INT32_C(           0) },
    { -INT32_C(   151062852),
      -INT32_C(          28),
      -INT32_C(           1) },
    {  INT32_C(  1218639051),
      -INT32_C(          28),
       INT32_C(           5) },
    { -INT32_C(   690345980),
      -INT32_C(          27),
      -INT32_C(           5) },
    { -INT32_C(  1463689207),
      -INT32_C(          27),
      -INT32_C(          11) },
    { -INT32_C(   227686834),
      -INT32_C(          26),
      -INT32_C(           3) },
    { -INT32_C(  1135570502),
      -INT32_C(          26),
      -INT32_C(          17) },
    {  INT32_C(   914800967),
      -INT32_C(          25),
       INT32_C(          27) },
    {  INT32_C(   255794002),
      -INT32_C(          25),
       INT32_C(           8) },
    { -INT32_C(   586858990),
      -INT32_C(          24),
      -INT32_C(          35) },
    {  INT32_C(   908502834),
      -INT32_C(          24),
       INT32_C(          54) },
    { -INT32_C(   670236465),
      -INT32_C(          23),
      -INT32_C(          80) },
    {  INT32_C(   780193504),
      -INT32_C(          23),
       INT32_C(          93) },
    {  INT32_C(  1310846612),
      -INT32_C(          22),
       INT32_C(         313) },
    { -INT32_C(   972394113),
      -INT32_C(          22),
      -INT32_C(         232) },
    { -INT32_C(  2047045582),
      -INT32_C(          21),
      -INT32_C(         976) },
    { -INT32_C(  1114358869),
      -INT32_C(          21),
      -INT32_C(         531) },
    { -INT32_C(  1415865991),
      -INT32_C(          20),
      -INT32_C(        1350) },
    {  INT32_C(   266518848),
      -INT32_C(          20),
       INT32_C(         254) },
    { -INT32_C(  1578569791),
      -INT32_C(          19),
      -INT32_C(        3011) },
    {  INT32_C(  1372547261),
      -INT32_C(          19),
       INT32_C(        2618) },
    { -INT32_C(   710938538),
      -INT32_C(          18),
      -INT32_C(        2712) },
    { -INT32_C(  1801737887),
      -INT32_C(          18),
      -INT32_C(        6873) },
    { -INT32_C(   468084935),
      -INT32_C(          17),
      -INT32_C(        3571) },
    {  INT32_C(  1285729747),
      -INT32_C(          17),
       INT32_C(        9809) },
    { -INT32_C(  2030551738),
      -INT32_C(          16),
      -INT32_C(       30984) },
    { -INT32_C(  1080698114),
      -INT32_C(          16),
      -INT32_C(       16490) },
    { -INT32_C(  2040496695),
      -INT32_C(          15),
      -INT32_C(       62271) },
    {  INT32_C(  1020735462),
      -INT32_C(          15),
       INT32_C(       31150) },
    { -INT32_C(  2129496544),
      -INT32_C(          14),
      -INT32_C(      129974) },
    {  INT32_C(  1477815583),
      -INT32_C(          14),
       INT32_C(       90199) },
    {  INT32_C(   406597189),
      -INT32_C(          13),
       INT32_C(       49633) },
    {  INT32_C(   560258779),
      -INT32_C(          13),
       INT32_C(       68391) },
    {  INT32_C(   430398491),
      -INT32_C(          12),
       INT32_C(      105078) },
    { -INT32_C(     2605770),
      -INT32_C(          12),
      -INT32_C(         636) },
    { -INT32_C(  1601881926),
      -INT32_C(          11),
      -INT32_C(      782169) },
    { -INT32_C(  1998758808),
      -INT32_C(          11),
      -INT32_C(      975956) },
    { -INT32_C(   251007022),
      -INT32_C(          10),
      -INT32_C(      245124) },
    { -INT32_C(   515301476),
      -INT32_C(          10),
      -INT32_C(      503224) },
    {  INT32_C(   704218701),
      -INT32_C(           9),
       INT32_C(     1375427) },
    { -INT32_C(  2142609820),
      -INT32_C(           9),
      -INT32_C(     4184785) },
    { -INT32_C(   241569094),
      -INT32_C(           8),
      -INT32_C(      943629) },
    { -INT32_C(   370118097),
      -INT32_C(           8),
      -INT32_C(     1445774) },
    {  INT32_C(   311064234),
      -INT32_C(           7),
       INT32_C(     2430189) },
    { -INT32_C(  1516607534),
      -INT32_C(           7),
      -INT32_C(    11848496) },
    { -INT32_C(   225008554),
      -INT32_C(           6),
      -INT32_C(     3515759) },
    {  INT32_C(   282386627),
      -INT32_C(           6),
       INT32_C(     4412291) },
    { -INT32_C(   902181530),
      -INT32_C(           5),
      -INT32_C(    28193173) },
    { -INT32_C(   431324117),
      -INT32_C(           5),
      -INT32_C(    13478879) },
    { -INT32_C(  1512577930),
      -INT32_C(           4),
      -INT32_C(    94536121) },
    {  INT32_C(     9357142),
      -INT32_C(           4),
       INT32_C(      584821) },
    {  INT32_C(   269686845),
      -INT32_C(           3),
       INT32_C(    33710856) },
    { -INT32_C(   709513857),
      -INT32_C(           3),
      -INT32_C(    88689232) },
    {  INT32_C(   348670801),
      -INT32_C(           2),
       INT32_C(    87167700) },
    { -INT32_C(  1859806165),
      -INT32_C(           2),
      -INT32_C(   464951541) },
    { -INT32_C(  1789108631),
      -INT32_C(           1),
      -INT32_C(   894554315) },
    {  INT32_C(  1484498658),
      -INT32_C(           1),
       INT32_C(   742249329) },
    { -INT32_C(   520269174),
       INT32_C(           0),
      -INT32_C(   520269174) },
    {  INT32_C(  1474399257),
       INT32_C(           0),
       INT32_C(  1474399257) },
    {  INT32_C(   610792612),
       INT32_C(           0),
       INT32_C(   610792612) },
    { -INT32_C(   201778015),
       INT32_C(           1),
      -INT32_C(   403556030) },
    { -INT32_C(  1828208281),
       INT32_C(           1),
                   INT32_MIN },
    { -INT32_C(   953930659),
       INT32_C(           2),
                   INT32_MIN },
    {  INT32_C(  1834778763),
       INT32_C(           2),
                   INT32_MAX },
    { -INT32_C(  1312368857),
       INT32_C(           3),
                   INT32_MIN },
    {  INT32_C(  1116914473),
       INT32_C(           3),
                   INT32_MAX },
    { -INT32_C(   191270065),
       INT32_C(           4),
                   INT32_MIN },
    {  INT32_C(   135790695),
       INT32_C(           4),
                   INT32_MAX },
    { -INT32_C(  2063920868),
       INT32_C(           5),
                   INT32_MIN },
    {  INT32_C(   806814675),
       INT32_C(           5),
                   INT32_MAX },
    { -INT32_C(  1158202577),
       INT32_C(           6),
                   INT32_MIN },
    { -INT32_C(   483896388),
       INT32_C(           6),
                   INT32_MIN },
    {  INT32_C(  1402269226),
       INT32_C(           7),
                   INT32_MAX },
    {  INT32_C(    26617521),
       INT32_C(           7),
                   INT32_MAX },
    {  INT32_C(    16068505),
       INT32_C(           8),
                   INT32_MAX },
    {  INT32_C(  1275661616),
       INT32_C(           8),
                   INT32_MAX },
    { -INT32_C(   238025698),
       INT32_C(           9),
                   INT32_MIN },
    {  INT32_C(   925034247),
       INT32_C(           9),
                   INT32_MAX },
    { -INT32_C(   537847517),
       INT32_C(          10),
                   INT32_MIN },
    { -INT32_C(  1748887187),
       INT32_C(          10),
                   INT32_MIN },
    { -INT32_C(  1175759353),
       INT32_C(          11),
                   INT32_MIN },
    {  INT32_C(   381321597),
       INT32_C(          11),
                   INT32_MAX },
    { -INT32_C(   535318608),
       INT32_C(          12),
                   INT32_MIN },
    { -INT32_C(   634576708),
       INT32_C(          12),
                   INT32_MIN },
    {  INT32_C(   751631652),
       INT32_C(          13),
                   INT32_MAX },
    {  INT32_C(   140766949),
       INT32_C(          13),
                   INT32_MAX },
    {  INT32_C(  1961317383),
       INT32_C(          14),
                   INT32_MAX },
    {  INT32_C(  1963764078),
       INT32_C(          14),
                   INT32_MAX },
    {  INT32_C(  2083452927),
       INT32_C(          15),
                   INT32_MAX },
    {  INT32_C(   680781944),
       INT32_C(          15),
                   INT32_MAX },
    {  INT32_C(  1393142423),
       INT32_C(          16),
                   INT32_MAX },
    { -INT32_C(   298961206),
       INT32_C(          16),
                   INT32_MIN },
    {  INT32_C(   404421171),
       INT32_C(          17),
                   INT32_MAX },
    { -INT32_C(   283083288),
       INT32_C(          17),
                   INT32_MIN },
    {  INT32_C(  1080297426),
       INT32_C(          18),
                   INT32_MAX },
    { -INT32_C(  1330286416),
       INT32_C(          18),
                   INT32_MIN },
    { -INT32_C(   550706073),
       INT32_C(          19),
                   INT32_MIN },
    {  INT32_C(  1678229452),
       INT32_C(          19),
                   INT32_MAX },
    {  INT32_C(   867635305),
       INT32_C(          20),
                   INT32_MAX },
    {  INT32_C(  2049107270),
       INT32_C(          20),
                   INT32_MAX },
    { -INT32_C(   946717473),
       INT32_C(          21),
                   INT32_MIN },
    { -INT32_C(  1934117958),
       INT32_C(          21),
                   INT32_MIN },
    {  INT32_C(  1808538554),
       INT32_C(          22),
                   INT32_MAX },
    { -INT32_C(   233078389),
       INT32_C(          22),
                   INT32_MIN },
    {  INT32_C(   852576101),
       INT32_C(          23),
                   INT32_MAX },
    {  INT32_C(  1888933895),
       INT32_C(          23),
                   INT32_MAX },
    {  INT32_C(   799297001),
       INT32_C(          24),
                   INT32_MAX },
    {  INT32_C(   313116211),
       INT32_C(          24),
                   INT32_MAX },
    { -INT32_C(  1126548478),
       INT32_C(          25),
                   INT32_MIN },
    { -INT32_C(  1454861841),
       INT32_C(          25),
                   INT32_MIN },
    {  INT32_C(   924062892),
       INT32_C(          26),
                   INT32_MAX },
    { -INT32_C(    81186922),
       INT32_C(          26),
                   INT32_MIN },
    {  INT32_C(  2116942455),
       INT32_C(          27),
                   INT32_MAX },
    { -INT32_C(  1141980206),
       INT32_C(          27),
                   INT32_MIN },
    {  INT32_C(  1156289041),
       INT32_C(          28),
                   INT32_MAX },
    {  INT32_C(  1532400728),
       INT32_C(          28),
                   INT32_MAX },
    { -INT32_C(  1088999216),
       INT32_C(          29),
                   INT32_MIN },
    {  INT32_C(  1835622593),
       INT32_C(          29),
                   INT32_MAX },
    {  INT32_C(   178552180),
       INT32_C(          30),
                   INT32_MAX },
    {  INT32_C(   604425645),
       INT32_C(          30),
                   INT32_MAX },
    { -INT32_C(  1700645945),
       INT32_C(          31),
                   INT32_MIN },
    {  INT32_C(   139825399),
       INT32_C(          31),
                   INT32_MAX },
    {  INT32_C(  2068594723),
       INT32_C(          32),
                   INT32_MAX },
    { -INT32_C(  1512660267),
       INT32_C(          32),
                   INT32_MIN },
    { -INT32_C(  1805259053),
       INT32_C(          33),
                   INT32_MIN },
    { -INT32_C(  1040003506),
       INT32_C(          33),
                   INT32_MIN },
    { -INT32_C(   120740277),
       INT32_C(          34),
                   INT32_MIN },
    {  INT32_C(   991744884),
       INT32_C(          34),
                   INT32_MAX },
    { -INT32_C(    36323834),
       INT32_C(          35),
                   INT32_MIN },
    {  INT32_C(  1912941391),
       INT32_C(          35),
                   INT32_MAX },
    {  INT32_C(  1089294699),
       INT32_C(          36),
                   INT32_MAX },
    { -INT32_C(   941177612),
       INT32_C(          36),
                   INT32_MIN },
    {  INT32_C(     5983154),
       INT32_C(          37),
                   INT32_MAX },
    {  INT32_C(  1690459417),
       INT32_C(          37),
                   INT32_MAX },
    {  INT32_C(  2019397380),
       INT32_C(          38),
                   INT32_MAX },
    {  INT32_C(  1773369698),
       INT32_C(          38),
                   INT32_MAX },
    { -INT32_C(  2023323336),
       INT32_C(          39),
                   INT32_MIN },
    {  INT32_C(   536439988),
       INT32_C(          39),
                   INT32_MAX },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    int32_t r = simde_vqrshls_s32(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_i32(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    int32_t a = simde_test_codegen_random_i32();
    int32_t b = simde_test_codegen_random_i32();
    int32_t r = simde_vqrshls_s32(a, b);

    simde_test_codegen_write_i32(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -80 ; i < 80 ; i++) {
    int32_t a = simde_test_codegen_random_i32();
    int32_t b = i / 2;
    int32_t r = simde_vqrshls_s32(a, b);

    simde_test_codegen_write_i32(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshld_s64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    int64_t a;
    int64_t b;
    int64_t r;
  } test_vec[] = {
    {  INT64_C( 4440879034276206119),
      -INT64_C( 6294382681135318368),
       INT64_C(                   0) },
    { -INT64_C( 8749732448969590841),
      -INT64_C( 6900581735583010771),
                           INT64_MIN },
    { -INT64_C( 8449822818109964405),
      -INT64_C( 1215652871492777172),
                           INT64_MIN },
    {  INT64_C( 8633853803546449561),
      -INT64_C( 7465387029608343732),
                           INT64_MAX },
    {  INT64_C( 6040122215907194205),
       INT64_C(  735755184919826206),
                           INT64_MAX },
    {  INT64_C( 2901768264706506627),
      -INT64_C( 6583890719471195907),
       INT64_C(  362721033088313328) },
    { -INT64_C( 6141857055932434360),
      -INT64_C(   34017255562899700),
                           INT64_MIN },
    { -INT64_C(   42481821068436379),
      -INT64_C( 1034232896568076357),
       INT64_C(                   0) },
    {  INT64_C( 6600346650330370403),
      -INT64_C(                  70),
       INT64_C(                   0) },
    {  INT64_C( 1454206035225512627),
      -INT64_C(                  69),
       INT64_C(                   0) },
    { -INT64_C( 3873448557284033476),
      -INT64_C(                  68),
       INT64_C(                   0) },
    {  INT64_C( 6174841851515455254),
      -INT64_C(                  67),
       INT64_C(                   0) },
    { -INT64_C( 1608949216510209534),
      -INT64_C(                  66),
       INT64_C(                   0) },
    {  INT64_C( 2318024378666222695),
      -INT64_C(                  65),
       INT64_C(                   0) },
    {  INT64_C( 6204726910713299266),
      -INT64_C(                  64),
       INT64_C(                   0) },
    {  INT64_C( 6497786447256741861),
      -INT64_C(                  63),
       INT64_C(                   1) },
    { -INT64_C( 5334700340343574312),
      -INT64_C(                  62),
      -INT64_C(                   1) },
    { -INT64_C( 7431229283662819366),
      -INT64_C(                  61),
      -INT64_C(                   3) },
    { -INT64_C( 4535399854168719751),
      -INT64_C(                  60),
      -INT64_C(                   4) },
    { -INT64_C( 4350822575734446868),
      -INT64_C(                  59),
      -INT64_C(                   8) },
    {  INT64_C( 8735234639227144502),
      -INT64_C(                  58),
       INT64_C(                  30) },
    {  INT64_C( 8630891392336856793),
      -INT64_C(                  57),
       INT64_C(                  60) },
    { -INT64_C( 6633450720136744621),
      -INT64_C(                  56),
      -INT64_C(                  92) },
    { -INT64_C( 8857869841899943325),
      -INT64_C(                  55),
      -INT64_C(                 246) },
    { -INT64_C( 8725931340852501297),
      -INT64_C(                  54),
      -INT64_C(                 484) },
    { -INT64_C( 8665350969519432119),
      -INT64_C(                  53),
      -INT64_C(                 962) },
    { -INT64_C( 8031703545357537746),
      -INT64_C(                  52),
      -INT64_C(                1783) },
    { -INT64_C( 4105046489986797262),
      -INT64_C(                  51),
      -INT64_C(                1823) },
    { -INT64_C( 7203193884498390827),
      -INT64_C(                  50),
      -INT64_C(                6398) },
    {  INT64_C( 7707806563920231995),
      -INT64_C(                  49),
       INT64_C(               13692) },
    { -INT64_C( 8624999844306506097),
      -INT64_C(                  48),
      -INT64_C(               30642) },
    { -INT64_C( 1006536798886089655),
      -INT64_C(                  47),
      -INT64_C(                7152) },
    { -INT64_C( 7633182065961976543),
      -INT64_C(                  46),
      -INT64_C(              108474) },
    {  INT64_C( 2729740572335404092),
      -INT64_C(                  45),
       INT64_C(               77584) },
    {  INT64_C( 6019112617696702890),
      -INT64_C(                  44),
       INT64_C(              342147) },
    { -INT64_C( 4365504667160636837),
      -INT64_C(                  43),
      -INT64_C(              496300) },
    { -INT64_C( 7437051174286991655),
      -INT64_C(                  42),
      -INT64_C(             1690990) },
    {  INT64_C( 8573149227693100861),
      -INT64_C(                  41),
       INT64_C(             3898617) },
    {  INT64_C( 7313563264040295202),
      -INT64_C(                  40),
       INT64_C(             6651647) },
    { -INT64_C( 8577308151767035247),
      -INT64_C(                  39),
      -INT64_C(            15602033) },
    { -INT64_C( 5325077232258109659),
      -INT64_C(                  38),
      -INT64_C(            19372518) },
    {  INT64_C( 4366382887479093589),
      -INT64_C(                  37),
       INT64_C(            31769617) },
    { -INT64_C( 6859855182853370019),
      -INT64_C(                  36),
      -INT64_C(            99824031) },
    {  INT64_C( 1598720627208137208),
      -INT64_C(                  35),
       INT64_C(            46528894) },
    { -INT64_C( 4167604879325926405),
      -INT64_C(                  34),
      -INT64_C(           242586532) },
    { -INT64_C( 7840625982021520422),
      -INT64_C(                  33),
      -INT64_C(           912768997) },
    {  INT64_C( 8520185412390131599),
      -INT64_C(                  32),
       INT64_C(          1983760254) },
    { -INT64_C( 2848723725587771024),
      -INT64_C(                  31),
      -INT64_C(          1326540357) },
    { -INT64_C( 7390840472445710793),
      -INT64_C(                  30),
      -INT64_C(          6883256577) },
    {  INT64_C( 7345579720631473704),
      -INT64_C(                  29),
       INT64_C(         13682208435) },
    {  INT64_C( 2470295532384545621),
      -INT64_C(                  28),
       INT64_C(          9202567981) },
    { -INT64_C( 1727136293160000174),
      -INT64_C(                  27),
      -INT64_C(         12868168154) },
    { -INT64_C( 1197113317398654455),
      -INT64_C(                  26),
      -INT64_C(         17838378510) },
    {  INT64_C(  184996873680842581),
      -INT64_C(                  25),
       INT64_C(          5513336470) },
    {  INT64_C( 2881492140923589672),
      -INT64_C(                  24),
       INT64_C(        171750315483) },
    { -INT64_C( 8927157577614720894),
      -INT64_C(                  23),
      -INT64_C(       1064200112535) },
    {  INT64_C(  100960097421536687),
      -INT64_C(                  22),
       INT64_C(         24070762973) },
    {  INT64_C( 1823324467224228766),
      -INT64_C(                  21),
       INT64_C(        869428857433) },
    { -INT64_C( 4517126651951963821),
      -INT64_C(                  20),
      -INT64_C(       4307867671921) },
    { -INT64_C( 5588948433303657103),
      -INT64_C(                  19),
      -INT64_C(      10660073153121) },
    {  INT64_C(  679075335331495358),
      -INT64_C(                  18),
       INT64_C(       2590466824842) },
    { -INT64_C( 6030986925772072637),
      -INT64_C(                  17),
      -INT64_C(      46012778669526) },
    { -INT64_C( 6574652708809850429),
      -INT64_C(                  16),
      -INT64_C(     100321238842924) },
    { -INT64_C( 8683501065291647031),
      -INT64_C(                  15),
      -INT64_C(     264999422158559) },
    {  INT64_C( 8733427689126537150),
      -INT64_C(                  14),
       INT64_C(     533046123603915) },
    { -INT64_C( 1369944031761692681),
      -INT64_C(                  13),
      -INT64_C(     167229496064660) },
    { -INT64_C( 4750258618049376196),
      -INT64_C(                  12),
      -INT64_C(    1159731107922211) },
    {  INT64_C( 7113445112713749351),
      -INT64_C(                  11),
       INT64_C(    3473361871442260) },
    { -INT64_C(   45107426318570833),
      -INT64_C(                  10),
      -INT64_C(      44050221014229) },
    {  INT64_C( 4925613112764075078),
      -INT64_C(                   9),
       INT64_C(    9620338110867334) },
    {  INT64_C( 6975731645872818460),
      -INT64_C(                   8),
       INT64_C(   27248951741690697) },
    {  INT64_C( 1112469784870606953),
      -INT64_C(                   7),
       INT64_C(    8691170194301617) },
    {  INT64_C(  214577098616339699),
      -INT64_C(                   6),
       INT64_C(    3352767165880308) },
    { -INT64_C( 5643910191415997856),
      -INT64_C(                   5),
      -INT64_C(  176372193481749933) },
    { -INT64_C( 8183163509056315626),
      -INT64_C(                   4),
      -INT64_C(  511447719316019727) },
    {  INT64_C( 4348521523621658857),
      -INT64_C(                   3),
       INT64_C(  543565190452707357) },
    {  INT64_C(  200184383116103177),
      -INT64_C(                   2),
       INT64_C(   50046095779025794) },
    {  INT64_C( 2000466870505453692),
      -INT64_C(                   1),
       INT64_C( 1000233435252726846) },
    { -INT64_C(  132405296459250315),
       INT64_C(                   0),
      -INT64_C(  132405296459250315) },
    { -INT64_C( 2584888258965486182),
       INT64_C(                   1),
      -INT64_C( 5169776517930972364) },
    {  INT64_C(  473837330162079983),
       INT64_C(                   2),
       INT64_C( 1895349320648319932) },
    { -INT64_C( 8448231643749548384),
       INT64_C(                   3),
                           INT64_MIN },
    {  INT64_C( 2338297871421729887),
       INT64_C(                   4),
                           INT64_MAX },
    {  INT64_C( 5458685304259263251),
       INT64_C(                   5),
                           INT64_MAX },
    {  INT64_C( 7713945749773116029),
       INT64_C(                   6),
                           INT64_MAX },
    {  INT64_C( 6777023534126408814),
       INT64_C(                   7),
                           INT64_MAX },
    { -INT64_C( 7969885357916081730),
       INT64_C(                   8),
                           INT64_MIN },
    {  INT64_C(  840546464596191490),
       INT64_C(                   9),
                           INT64_MAX },
    {  INT64_C( 7336398940399634331),
       INT64_C(                  10),
                           INT64_MAX },
    {  INT64_C( 8260152246644760900),
       INT64_C(                  11),
                           INT64_MAX },
    { -INT64_C( 4740851436211440347),
       INT64_C(                  12),
                           INT64_MIN },
    {  INT64_C( 9160261049010465925),
       INT64_C(                  13),
                           INT64_MAX },
    { -INT64_C( 7661830980354362801),
       INT64_C(                  14),
                           INT64_MIN },
    { -INT64_C( 6071562005408869737),
       INT64_C(                  15),
                           INT64_MIN },
    {  INT64_C( 1614967698273301743),
       INT64_C(                  16),
                           INT64_MAX },
    { -INT64_C( 7019905315174185168),
       INT64_C(                  17),
                           INT64_MIN },
    { -INT64_C( 4223656069334533765),
       INT64_C(                  18),
                           INT64_MIN },
    { -INT64_C( 6605708948818347231),
       INT64_C(                  19),
                           INT64_MIN },
    {  INT64_C( 6262739233728864834),
       INT64_C(                  20),
                           INT64_MAX },
    { -INT64_C( 2869638089550392513),
       INT64_C(                  21),
                           INT64_MIN },
    { -INT64_C( 4509220744627679112),
       INT64_C(                  22),
                           INT64_MIN },
    { -INT64_C( 8242787662807073726),
       INT64_C(                  23),
                           INT64_MIN },
    { -INT64_C( 2099722359595275681),
       INT64_C(                  24),
                           INT64_MIN },
    { -INT64_C( 4715095882789206180),
       INT64_C(                  25),
                           INT64_MIN },
    {  INT64_C( 9155484054200792494),
       INT64_C(                  26),
                           INT64_MAX },
    { -INT64_C( 7844124633883773107),
       INT64_C(                  27),
                           INT64_MIN },
    {  INT64_C( 2991222266518865149),
       INT64_C(                  28),
                           INT64_MAX },
    {  INT64_C( 4557487317197189947),
       INT64_C(                  29),
                           INT64_MAX },
    { -INT64_C( 4511310000348683096),
       INT64_C(                  30),
                           INT64_MIN },
    { -INT64_C( 1451990942514291950),
       INT64_C(                  31),
                           INT64_MIN },
    {  INT64_C( 2520060751466197506),
       INT64_C(                  32),
                           INT64_MAX },
    { -INT64_C(  733555341972481279),
       INT64_C(                  33),
                           INT64_MIN },
    {  INT64_C(  845902643625352583),
       INT64_C(                  34),
                           INT64_MAX },
    { -INT64_C( 7641716232217881062),
       INT64_C(                  35),
                           INT64_MIN },
    { -INT64_C( 8276104784778002915),
       INT64_C(                  36),
                           INT64_MIN },
    {  INT64_C( 3710104816821717305),
       INT64_C(                  37),
                           INT64_MAX },
    { -INT64_C( 5741549023560575411),
       INT64_C(                  38),
                           INT64_MIN },
    { -INT64_C( 1573255120832388607),
       INT64_C(                  39),
                           INT64_MIN },
    {  INT64_C( 2757069308578139946),
       INT64_C(                  40),
                           INT64_MAX },
    { -INT64_C( 6864847772539067684),
       INT64_C(                  41),
                           INT64_MIN },
    { -INT64_C( 2898849396808029569),
       INT64_C(                  42),
                           INT64_MIN },
    {  INT64_C( 6093816876455722457),
       INT64_C(                  43),
                           INT64_MAX },
    {  INT64_C( 8014804880445832998),
       INT64_C(                  44),
                           INT64_MAX },
    { -INT64_C( 8096430823714527312),
       INT64_C(                  45),
                           INT64_MIN },
    { -INT64_C( 7400986762335188579),
       INT64_C(                  46),
                           INT64_MIN },
    {  INT64_C( 8242944442126874343),
       INT64_C(                  47),
                           INT64_MAX },
    {  INT64_C( 7878133569993207633),
       INT64_C(                  48),
                           INT64_MAX },
    { -INT64_C( 7113976572369941337),
       INT64_C(                  49),
                           INT64_MIN },
    {  INT64_C( 4320740210876935068),
       INT64_C(                  50),
                           INT64_MAX },
    {  INT64_C( 6543642333547345424),
       INT64_C(                  51),
                           INT64_MAX },
    { -INT64_C( 4878926623715470049),
       INT64_C(                  52),
                           INT64_MIN },
    {  INT64_C( 7672930688302299870),
       INT64_C(                  53),
                           INT64_MAX },
    { -INT64_C(  777788040192488540),
       INT64_C(                  54),
                           INT64_MIN },
    {  INT64_C( 5788849770090037491),
       INT64_C(                  55),
                           INT64_MAX },
    {  INT64_C(  380122230265058421),
       INT64_C(                  56),
                           INT64_MAX },
    {  INT64_C( 1687264862711036209),
       INT64_C(                  57),
                           INT64_MAX },
    { -INT64_C( 2178820902755487255),
       INT64_C(                  58),
                           INT64_MIN },
    { -INT64_C( 6104854463009450555),
       INT64_C(                  59),
                           INT64_MIN },
    {  INT64_C( 2086429880151798984),
       INT64_C(                  60),
                           INT64_MAX },
    {  INT64_C( 2360033774048958517),
       INT64_C(                  61),
                           INT64_MAX },
    { -INT64_C( 2628217526476156745),
       INT64_C(                  62),
                           INT64_MIN },
    {  INT64_C( 6892504795774543047),
       INT64_C(                  63),
                           INT64_MAX },
    { -INT64_C( 8172833372237800052),
       INT64_C(                  64),
                           INT64_MIN },
    { -INT64_C( 1337015801689427941),
       INT64_C(                  65),
                           INT64_MIN },
    { -INT64_C( 2758428302458593081),
       INT64_C(                  66),
                           INT64_MIN },
    {  INT64_C( 3347157083154123132),
       INT64_C(                  67),
                           INT64_MAX },
    {  INT64_C( 2324202678863184474),
       INT64_C(                  68),
                           INT64_MAX },
    {  INT64_C( 3967076913208457450),
       INT64_C(                  69),
                           INT64_MAX },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    int64_t r = simde_vqrshld_s64(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_i64(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    int64_t a = simde_test_codegen_random_i64();
    int64_t b = simde_test_codegen_random_i64();
    int64_t r = simde_vqrshld_s64(a, b);

    simde_test_codegen_write_i64(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i64(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -70 ; i < 70 ; i++) {
    int64_t a = simde_test_codegen_random_i64();
    int64_t b = i;
    int64_t r = simde_vqrshld_s64(a, b);

    simde_test_codegen_write_i64(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i64(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlb_u8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    uint8_t a;
    int8_t b;
    uint8_t r;
  } test_vec[] = {
    { UINT8_C(230),
      -INT8_C( 102),
      UINT8_C(  0) },
    { UINT8_C( 28),
      -INT8_C(  91),
      UINT8_C(  0) },
    { UINT8_C(199),
           INT8_MIN,
      UINT8_C(  0) },
    { UINT8_C( 97),
       INT8_C( 125),
         UINT8_MAX },
    { UINT8_C(164),
       INT8_C(  97),
         UINT8_MAX },
    { UINT8_C( 92),
       INT8_C( 124),
         UINT8_MAX },
    { UINT8_C(224),
           INT8_MAX,
         UINT8_MAX },
    { UINT8_C( 35),
       INT8_C(  32),
         UINT8_MAX },
    { UINT8_C( 85),
      -INT8_C(  10),
      UINT8_C(  0) },
    { UINT8_C( 32),
      -INT8_C(   9),
      UINT8_C(  0) },
    { UINT8_C(249),
      -INT8_C(   9),
      UINT8_C(  0) },
    { UINT8_C(143),
      -INT8_C(   9),
      UINT8_C(  0) },
    { UINT8_C(231),
      -INT8_C(   9),
      UINT8_C(  0) },
    { UINT8_C( 20),
      -INT8_C(   8),
      UINT8_C(  0) },
    { UINT8_C(153),
      -INT8_C(   8),
      UINT8_C(  1) },
    { UINT8_C( 97),
      -INT8_C(   8),
      UINT8_C(  0) },
    { UINT8_C(  5),
      -INT8_C(   8),
      UINT8_C(  0) },
    { UINT8_C(  7),
      -INT8_C(   7),
      UINT8_C(  0) },
    { UINT8_C( 22),
      -INT8_C(   7),
      UINT8_C(  0) },
    { UINT8_C( 24),
      -INT8_C(   7),
      UINT8_C(  0) },
    { UINT8_C( 42),
      -INT8_C(   7),
      UINT8_C(  0) },
    { UINT8_C(138),
      -INT8_C(   6),
      UINT8_C(  2) },
    { UINT8_C(187),
      -INT8_C(   6),
      UINT8_C(  3) },
    { UINT8_C( 16),
      -INT8_C(   6),
      UINT8_C(  0) },
    { UINT8_C( 37),
      -INT8_C(   6),
      UINT8_C(  1) },
    { UINT8_C(216),
      -INT8_C(   5),
      UINT8_C(  7) },
    { UINT8_C(182),
      -INT8_C(   5),
      UINT8_C(  6) },
    { UINT8_C(236),
      -INT8_C(   5),
      UINT8_C(  7) },
    { UINT8_C( 88),
      -INT8_C(   5),
      UINT8_C(  3) },
    { UINT8_C( 23),
      -INT8_C(   4),
      UINT8_C(  1) },
    { UINT8_C(105),
      -INT8_C(   4),
      UINT8_C(  7) },
    { UINT8_C(252),
      -INT8_C(   4),
      UINT8_C( 16) },
    { UINT8_C(120),
      -INT8_C(   4),
      UINT8_C(  8) },
    { UINT8_C(198),
      -INT8_C(   3),
      UINT8_C( 25) },
    { UINT8_C(121),
      -INT8_C(   3),
      UINT8_C( 15) },
    { UINT8_C( 89),
      -INT8_C(   3),
      UINT8_C( 11) },
    { UINT8_C( 69),
      -INT8_C(   3),
      UINT8_C(  9) },
    { UINT8_C(156),
      -INT8_C(   2),
      UINT8_C( 39) },
    { UINT8_C(121),
      -INT8_C(   2),
      UINT8_C( 30) },
    { UINT8_C(155),
      -INT8_C(   2),
      UINT8_C( 39) },
    { UINT8_C(188),
      -INT8_C(   2),
      UINT8_C( 47) },
    { UINT8_C(114),
      -INT8_C(   1),
      UINT8_C( 57) },
    { UINT8_C( 42),
      -INT8_C(   1),
      UINT8_C( 21) },
    { UINT8_C(163),
      -INT8_C(   1),
      UINT8_C( 82) },
    { UINT8_C(134),
      -INT8_C(   1),
      UINT8_C( 67) },
    { UINT8_C(195),
       INT8_C(   0),
      UINT8_C(195) },
    { UINT8_C(  5),
       INT8_C(   0),
      UINT8_C(  5) },
    { UINT8_C(139),
       INT8_C(   0),
      UINT8_C(139) },
    { UINT8_C(203),
       INT8_C(   0),
      UINT8_C(203) },
    { UINT8_C( 27),
       INT8_C(   0),
      UINT8_C( 27) },
    { UINT8_C(164),
       INT8_C(   0),
      UINT8_C(164) },
    { UINT8_C(245),
       INT8_C(   0),
      UINT8_C(245) },
    { UINT8_C(165),
       INT8_C(   1),
         UINT8_MAX },
    { UINT8_C( 95),
       INT8_C(   1),
      UINT8_C(190) },
    { UINT8_C(  5),
       INT8_C(   1),
      UINT8_C( 10) },
    { UINT8_C(202),
       INT8_C(   1),
         UINT8_MAX },
    { UINT8_C( 55),
       INT8_C(   2),
      UINT8_C(220) },
    { UINT8_C(187),
       INT8_C(   2),
         UINT8_MAX },
    { UINT8_C(183),
       INT8_C(   2),
         UINT8_MAX },
    { UINT8_C(144),
       INT8_C(   2),
         UINT8_MAX },
    { UINT8_C(211),
       INT8_C(   3),
         UINT8_MAX },
    { UINT8_C( 32),
       INT8_C(   3),
         UINT8_MAX },
    { UINT8_C(140),
       INT8_C(   3),
         UINT8_MAX },
    { UINT8_C( 75),
       INT8_C(   3),
         UINT8_MAX },
    { UINT8_C(230),
       INT8_C(   4),
         UINT8_MAX },
    { UINT8_C(  5),
       INT8_C(   4),
      UINT8_C( 80) },
    { UINT8_C(164),
       INT8_C(   4),
         UINT8_MAX },
    { UINT8_C( 44),
       INT8_C(   4),
         UINT8_MAX },
    { UINT8_C(161),
       INT8_C(   5),
         UINT8_MAX },
    { UINT8_C( 29),
       INT8_C(   5),
         UINT8_MAX },
    { UINT8_C(199),
       INT8_C(   5),
         UINT8_MAX },
    { UINT8_C( 94),
       INT8_C(   5),
         UINT8_MAX },
    { UINT8_C(144),
       INT8_C(   6),
         UINT8_MAX },
    { UINT8_C(241),
       INT8_C(   6),
         UINT8_MAX },
    { UINT8_C(  1),
       INT8_C(   6),
      UINT8_C( 64) },
    { UINT8_C( 22),
       INT8_C(   6),
         UINT8_MAX },
    { UINT8_C(180),
       INT8_C(   7),
         UINT8_MAX },
    { UINT8_C(  6),
       INT8_C(   7),
         UINT8_MAX },
    { UINT8_C(162),
       INT8_C(   7),
         UINT8_MAX },
    { UINT8_C(127),
       INT8_C(   7),
         UINT8_MAX },
    { UINT8_C( 33),
       INT8_C(   8),
         UINT8_MAX },
    { UINT8_C( 70),
       INT8_C(   8),
         UINT8_MAX },
    { UINT8_C(116),
       INT8_C(   8),
         UINT8_MAX },
    { UINT8_C(199),
       INT8_C(   8),
         UINT8_MAX },
    { UINT8_C(165),
       INT8_C(   9),
         UINT8_MAX },
    { UINT8_C(122),
       INT8_C(   9),
         UINT8_MAX },
    { UINT8_C(145),
       INT8_C(   9),
         UINT8_MAX },
    { UINT8_C(221),
       INT8_C(   9),
         UINT8_MAX },
    { UINT8_C(  1),
      -INT8_C(  10),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   9),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   8),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   7),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   6),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   5),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   4),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   3),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   2),
      UINT8_C(  0) },
    { UINT8_C(  1),
      -INT8_C(   1),
      UINT8_C(  1) },
    { UINT8_C(  1),
       INT8_C(   0),
      UINT8_C(  1) },
    { UINT8_C(  1),
       INT8_C(   1),
      UINT8_C(  2) },
    { UINT8_C(  1),
       INT8_C(   2),
      UINT8_C(  4) },
    { UINT8_C(  1),
       INT8_C(   3),
      UINT8_C(  8) },
    { UINT8_C(  1),
       INT8_C(   4),
      UINT8_C( 16) },
    { UINT8_C(  1),
       INT8_C(   5),
      UINT8_C( 32) },
    { UINT8_C(  1),
       INT8_C(   6),
      UINT8_C( 64) },
    { UINT8_C(  1),
       INT8_C(   7),
      UINT8_C(128) },
    { UINT8_C(  1),
       INT8_C(   8),
         UINT8_MAX },
    { UINT8_C(  1),
       INT8_C(   9),
         UINT8_MAX },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    uint8_t r = simde_vqrshlb_u8(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_u8(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    uint8_t a = simde_test_codegen_random_u8();
    int8_t b = simde_test_codegen_random_i8();
    uint8_t r = simde_vqrshlb_u8(a, b);

    simde_test_codegen_write_u8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -40; i < 40 ; i++) {
    uint8_t a = simde_test_codegen_random_u8();
    int8_t b = i / 4;
    uint8_t r = simde_vqrshlb_u8(a, b);

    simde_test_codegen_write_u8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -10; i < 10 ; i++) {
    uint8_t a = 1;
    int8_t b = i;
    uint8_t r = simde_vqrshlb_u8(a, b);

    simde_test_codegen_write_u8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlh_u16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    uint16_t a;
    int16_t b;
    uint16_t r;
  } test_vec[] = {
    { UINT16_C( 9680),
      -INT16_C( 11216),
           UINT16_MAX },
    { UINT16_C(11523),
      -INT16_C( 22795),
      UINT16_C(    6) },
    { UINT16_C(60986),
       INT16_C(  6367),
      UINT16_C(    0) },
    { UINT16_C(14582),
      -INT16_C( 13512),
           UINT16_MAX },
    { UINT16_C(14838),
       INT16_C( 17219),
           UINT16_MAX },
    { UINT16_C(24173),
       INT16_C( 20594),
           UINT16_MAX },
    { UINT16_C(21801),
      -INT16_C( 29743),
      UINT16_C(    0) },
    { UINT16_C(49619),
      -INT16_C( 23808),
      UINT16_C(49619) },
    { UINT16_C(12519),
      -INT16_C(    20),
      UINT16_C(    0) },
    { UINT16_C(60024),
      -INT16_C(    19),
      UINT16_C(    0) },
    { UINT16_C(27997),
      -INT16_C(    19),
      UINT16_C(    0) },
    { UINT16_C(38800),
      -INT16_C(    18),
      UINT16_C(    0) },
    { UINT16_C(28507),
      -INT16_C(    18),
      UINT16_C(    0) },
    { UINT16_C(20912),
      -INT16_C(    17),
      UINT16_C(    0) },
    { UINT16_C(59560),
      -INT16_C(    17),
      UINT16_C(    0) },
    { UINT16_C(40476),
      -INT16_C(    16),
      UINT16_C(    1) },
    { UINT16_C(24609),
      -INT16_C(    16),
      UINT16_C(    0) },
    { UINT16_C(36578),
      -INT16_C(    15),
      UINT16_C(    1) },
    { UINT16_C(21694),
      -INT16_C(    15),
      UINT16_C(    1) },
    { UINT16_C(59358),
      -INT16_C(    14),
      UINT16_C(    4) },
    { UINT16_C(44970),
      -INT16_C(    14),
      UINT16_C(    3) },
    { UINT16_C(32114),
      -INT16_C(    13),
      UINT16_C(    4) },
    { UINT16_C(29297),
      -INT16_C(    13),
      UINT16_C(    4) },
    { UINT16_C(22561),
      -INT16_C(    12),
      UINT16_C(    6) },
    { UINT16_C(39330),
      -INT16_C(    12),
      UINT16_C(   10) },
    { UINT16_C(65346),
      -INT16_C(    11),
      UINT16_C(   32) },
    { UINT16_C(54022),
      -INT16_C(    11),
      UINT16_C(   26) },
    { UINT16_C(24982),
      -INT16_C(    10),
      UINT16_C(   24) },
    { UINT16_C(17986),
      -INT16_C(    10),
      UINT16_C(   18) },
    { UINT16_C(60083),
      -INT16_C(     9),
      UINT16_C(  117) },
    { UINT16_C(53038),
      -INT16_C(     9),
      UINT16_C(  104) },
    { UINT16_C(20361),
      -INT16_C(     8),
      UINT16_C(   80) },
    { UINT16_C(27439),
      -INT16_C(     8),
      UINT16_C(  107) },
    { UINT16_C(60894),
      -INT16_C(     7),
      UINT16_C(  476) },
    { UINT16_C(48319),
      -INT16_C(     7),
      UINT16_C(  377) },
    { UINT16_C(27092),
      -INT16_C(     6),
      UINT16_C(  423) },
    { UINT16_C(18028),
      -INT16_C(     6),
      UINT16_C(  282) },
    { UINT16_C(56807),
      -INT16_C(     5),
      UINT16_C( 1775) },
    { UINT16_C( 2232),
      -INT16_C(     5),
      UINT16_C(   70) },
    { UINT16_C(23093),
      -INT16_C(     4),
      UINT16_C( 1443) },
    { UINT16_C(30625),
      -INT16_C(     4),
      UINT16_C( 1914) },
    { UINT16_C(42841),
      -INT16_C(     3),
      UINT16_C( 5355) },
    { UINT16_C(61514),
      -INT16_C(     3),
      UINT16_C( 7689) },
    { UINT16_C(36104),
      -INT16_C(     2),
      UINT16_C( 9026) },
    { UINT16_C(47926),
      -INT16_C(     2),
      UINT16_C(11982) },
    { UINT16_C(25975),
      -INT16_C(     1),
      UINT16_C(12988) },
    { UINT16_C(  139),
      -INT16_C(     1),
      UINT16_C(   70) },
    { UINT16_C(47796),
       INT16_C(     0),
      UINT16_C(47796) },
    { UINT16_C(37483),
       INT16_C(     0),
      UINT16_C(37483) },
    { UINT16_C(11176),
       INT16_C(     0),
      UINT16_C(11176) },
    { UINT16_C(31823),
       INT16_C(     1),
      UINT16_C(63646) },
    { UINT16_C(48020),
       INT16_C(     1),
           UINT16_MAX },
    { UINT16_C(31683),
       INT16_C(     2),
           UINT16_MAX },
    { UINT16_C(31640),
       INT16_C(     2),
           UINT16_MAX },
    { UINT16_C(52611),
       INT16_C(     3),
           UINT16_MAX },
    { UINT16_C( 9430),
       INT16_C(     3),
           UINT16_MAX },
    { UINT16_C(12100),
       INT16_C(     4),
           UINT16_MAX },
    { UINT16_C(36811),
       INT16_C(     4),
           UINT16_MAX },
    { UINT16_C(54303),
       INT16_C(     5),
           UINT16_MAX },
    { UINT16_C(22044),
       INT16_C(     5),
           UINT16_MAX },
    { UINT16_C(37775),
       INT16_C(     6),
           UINT16_MAX },
    { UINT16_C( 6843),
       INT16_C(     6),
           UINT16_MAX },
    { UINT16_C(28564),
       INT16_C(     7),
           UINT16_MAX },
    { UINT16_C(65493),
       INT16_C(     7),
           UINT16_MAX },
    { UINT16_C(32002),
       INT16_C(     8),
           UINT16_MAX },
    { UINT16_C(20778),
       INT16_C(     8),
           UINT16_MAX },
    { UINT16_C(49145),
       INT16_C(     9),
           UINT16_MAX },
    { UINT16_C(48140),
       INT16_C(     9),
           UINT16_MAX },
    { UINT16_C(42042),
       INT16_C(    10),
           UINT16_MAX },
    { UINT16_C(48696),
       INT16_C(    10),
           UINT16_MAX },
    { UINT16_C( 3697),
       INT16_C(    11),
           UINT16_MAX },
    { UINT16_C(46562),
       INT16_C(    11),
           UINT16_MAX },
    { UINT16_C(44605),
       INT16_C(    12),
           UINT16_MAX },
    { UINT16_C(23876),
       INT16_C(    12),
           UINT16_MAX },
    { UINT16_C(24706),
       INT16_C(    13),
           UINT16_MAX },
    { UINT16_C( 4531),
       INT16_C(    13),
           UINT16_MAX },
    { UINT16_C(28404),
       INT16_C(    14),
           UINT16_MAX },
    { UINT16_C(34860),
       INT16_C(    14),
           UINT16_MAX },
    { UINT16_C(  477),
       INT16_C(    15),
           UINT16_MAX },
    { UINT16_C(57223),
       INT16_C(    15),
           UINT16_MAX },
    { UINT16_C(45694),
       INT16_C(    16),
           UINT16_MAX },
    { UINT16_C(30512),
       INT16_C(    16),
           UINT16_MAX },
    { UINT16_C(15473),
       INT16_C(    17),
           UINT16_MAX },
    { UINT16_C(43828),
       INT16_C(    17),
           UINT16_MAX },
    { UINT16_C(27872),
       INT16_C(    18),
           UINT16_MAX },
    { UINT16_C(20841),
       INT16_C(    18),
           UINT16_MAX },
    { UINT16_C(19578),
       INT16_C(    19),
           UINT16_MAX },
    { UINT16_C(46855),
       INT16_C(    19),
           UINT16_MAX },
    { UINT16_C(    1),
      -INT16_C(    20),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    19),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    18),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    17),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    16),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    15),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    14),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    13),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    12),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    11),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(    10),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     9),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     8),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     7),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     6),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     5),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     4),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     3),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     2),
      UINT16_C(    0) },
    { UINT16_C(    1),
      -INT16_C(     1),
      UINT16_C(    1) },
    { UINT16_C(    1),
       INT16_C(     0),
      UINT16_C(    1) },
    { UINT16_C(    1),
       INT16_C(     1),
      UINT16_C(    2) },
    { UINT16_C(    1),
       INT16_C(     2),
      UINT16_C(    4) },
    { UINT16_C(    1),
       INT16_C(     3),
      UINT16_C(    8) },
    { UINT16_C(    1),
       INT16_C(     4),
      UINT16_C(   16) },
    { UINT16_C(    1),
       INT16_C(     5),
      UINT16_C(   32) },
    { UINT16_C(    1),
       INT16_C(     6),
      UINT16_C(   64) },
    { UINT16_C(    1),
       INT16_C(     7),
      UINT16_C(  128) },
    { UINT16_C(    1),
       INT16_C(     8),
      UINT16_C(  256) },
    { UINT16_C(    1),
       INT16_C(     9),
      UINT16_C(  512) },
    { UINT16_C(    1),
       INT16_C(    10),
      UINT16_C( 1024) },
    { UINT16_C(    1),
       INT16_C(    11),
      UINT16_C( 2048) },
    { UINT16_C(    1),
       INT16_C(    12),
      UINT16_C( 4096) },
    { UINT16_C(    1),
       INT16_C(    13),
      UINT16_C( 8192) },
    { UINT16_C(    1),
       INT16_C(    14),
      UINT16_C(16384) },
    { UINT16_C(    1),
       INT16_C(    15),
      UINT16_C(32768) },
    { UINT16_C(    1),
       INT16_C(    16),
           UINT16_MAX },
    { UINT16_C(    1),
       INT16_C(    17),
           UINT16_MAX },
    { UINT16_C(    1),
       INT16_C(    18),
           UINT16_MAX },
    { UINT16_C(    1),
       INT16_C(    19),
           UINT16_MAX },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    uint16_t r = simde_vqrshlh_u16(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_u16(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    uint16_t a = simde_test_codegen_random_u16();
    int16_t b = simde_test_codegen_random_i16();
    uint16_t r = simde_vqrshlh_u16(a, b);

    simde_test_codegen_write_u16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -40 ; i < 40 ; i++) {
    uint16_t a = simde_test_codegen_random_u16();
    int16_t b = i / 2;
    uint16_t r = simde_vqrshlh_u16(a, b);

    simde_test_codegen_write_u16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -20 ; i < 20 ; i++) {
    uint16_t a = 1;
    int16_t b = i;
    uint16_t r = simde_vqrshlh_u16(a, b);

    simde_test_codegen_write_u16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshls_u32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    uint32_t a;
    int32_t b;
    uint32_t r;
  } test_vec[] = {
    { UINT32_C(2090916658),
      -INT32_C(  1876413820),
      UINT32_C(         0) },
    { UINT32_C(1522271204),
      -INT32_C(  1655473296),
                UINT32_MAX },
    { UINT32_C(2975898342),
       INT32_C(   354288286),
      UINT32_C(         0) },
    { UINT32_C( 254363755),
       INT32_C(  1179680020),
                UINT32_MAX },
    { UINT32_C(3720540248),
       INT32_C(   107866914),
                UINT32_MAX },
    { UINT32_C(1583360494),
      -INT32_C(  1946438491),
      UINT32_C(         0) },
    { UINT32_C(3963444046),
      -INT32_C(   922592419),
                UINT32_MAX },
    { UINT32_C(3084397475),
       INT32_C(    16656552),
      UINT32_C(         0) },
    { UINT32_C( 987611160),
      -INT32_C(          40),
      UINT32_C(         0) },
    { UINT32_C(2571127467),
      -INT32_C(          39),
      UINT32_C(         0) },
    { UINT32_C( 435724660),
      -INT32_C(          39),
      UINT32_C(         0) },
    { UINT32_C(2745496405),
      -INT32_C(          38),
      UINT32_C(         0) },
    { UINT32_C(2895176015),
      -INT32_C(          38),
      UINT32_C(         0) },
    { UINT32_C(3765801533),
      -INT32_C(          37),
      UINT32_C(         0) },
    { UINT32_C(1704480189),
      -INT32_C(          37),
      UINT32_C(         0) },
    { UINT32_C(2372245109),
      -INT32_C(          36),
      UINT32_C(         0) },
    { UINT32_C(  46613334),
      -INT32_C(          36),
      UINT32_C(         0) },
    { UINT32_C(  26937485),
      -INT32_C(          35),
      UINT32_C(         0) },
    { UINT32_C(4263154601),
      -INT32_C(          35),
      UINT32_C(         0) },
    { UINT32_C(3600924551),
      -INT32_C(          34),
      UINT32_C(         0) },
    { UINT32_C(3716297120),
      -INT32_C(          34),
      UINT32_C(         0) },
    { UINT32_C(2159999171),
      -INT32_C(          33),
      UINT32_C(         0) },
    { UINT32_C(3152369221),
      -INT32_C(          33),
      UINT32_C(         0) },
    { UINT32_C(1112034284),
      -INT32_C(          32),
      UINT32_C(         0) },
    { UINT32_C( 457445518),
      -INT32_C(          32),
      UINT32_C(         0) },
    { UINT32_C(3239960600),
      -INT32_C(          31),
      UINT32_C(         2) },
    { UINT32_C(4206835571),
      -INT32_C(          31),
      UINT32_C(         2) },
    { UINT32_C(2547015926),
      -INT32_C(          30),
      UINT32_C(         2) },
    { UINT32_C(1433686930),
      -INT32_C(          30),
      UINT32_C(         1) },
    { UINT32_C(2429956683),
      -INT32_C(          29),
      UINT32_C(         5) },
    { UINT32_C(1951120264),
      -INT32_C(          29),
      UINT32_C(         4) },
    { UINT32_C(2495058950),
      -INT32_C(          28),
      UINT32_C(         9) },
    { UINT32_C(3165715364),
      -INT32_C(          28),
      UINT32_C(        12) },
    { UINT32_C(1333644763),
      -INT32_C(          27),
      UINT32_C(        10) },
    { UINT32_C(4215880708),
      -INT32_C(          27),
      UINT32_C(        31) },
    { UINT32_C( 781327004),
      -INT32_C(          26),
      UINT32_C(        12) },
    { UINT32_C(3095660141),
      -INT32_C(          26),
      UINT32_C(        46) },
    { UINT32_C(3242744377),
      -INT32_C(          25),
      UINT32_C(        97) },
    { UINT32_C( 473338901),
      -INT32_C(          25),
      UINT32_C(        14) },
    { UINT32_C(3434147112),
      -INT32_C(          24),
      UINT32_C(       205) },
    { UINT32_C(3297272040),
      -INT32_C(          24),
      UINT32_C(       197) },
    { UINT32_C( 840107309),
      -INT32_C(          23),
      UINT32_C(       100) },
    { UINT32_C(3710737473),
      -INT32_C(          23),
      UINT32_C(       442) },
    { UINT32_C(3809263478),
      -INT32_C(          22),
      UINT32_C(       908) },
    { UINT32_C(4271608005),
      -INT32_C(          22),
      UINT32_C(      1018) },
    { UINT32_C(4290831594),
      -INT32_C(          21),
      UINT32_C(      2046) },
    { UINT32_C(2686187128),
      -INT32_C(          21),
      UINT32_C(      1281) },
    { UINT32_C(3412905187),
      -INT32_C(          20),
      UINT32_C(      3255) },
    { UINT32_C(1519383596),
      -INT32_C(          20),
      UINT32_C(      1449) },
    { UINT32_C( 982295289),
      -INT32_C(          19),
      UINT32_C(      1874) },
    { UINT32_C(1964489215),
      -INT32_C(          19),
      UINT32_C(      3747) },
    { UINT32_C(1029251960),
      -INT32_C(          18),
      UINT32_C(      3926) },
    { UINT32_C(2638017715),
      -INT32_C(          18),
      UINT32_C(     10063) },
    { UINT32_C(1352531160),
      -INT32_C(          17),
      UINT32_C(     10319) },
    { UINT32_C(3589322994),
      -INT32_C(          17),
      UINT32_C(     27384) },
    { UINT32_C(2980076676),
      -INT32_C(          16),
      UINT32_C(     45472) },
    { UINT32_C(1225470032),
      -INT32_C(          16),
      UINT32_C(     18699) },
    { UINT32_C(3515062226),
      -INT32_C(          15),
      UINT32_C(    107271) },
    { UINT32_C(3360136016),
      -INT32_C(          15),
      UINT32_C(    102543) },
    { UINT32_C(1912971454),
      -INT32_C(          14),
      UINT32_C(    116759) },
    { UINT32_C(1829716372),
      -INT32_C(          14),
      UINT32_C(    111677) },
    { UINT32_C( 800959549),
      -INT32_C(          13),
      UINT32_C(     97773) },
    { UINT32_C(3909398117),
      -INT32_C(          13),
      UINT32_C(    477221) },
    { UINT32_C(1536861450),
      -INT32_C(          12),
      UINT32_C(    375210) },
    { UINT32_C(2812585429),
      -INT32_C(          12),
      UINT32_C(    686666) },
    { UINT32_C(2356750396),
      -INT32_C(          11),
      UINT32_C(   1150757) },
    { UINT32_C(2169815235),
      -INT32_C(          11),
      UINT32_C(   1059480) },
    { UINT32_C(4109589088),
      -INT32_C(          10),
      UINT32_C(   4013271) },
    { UINT32_C(3647013787),
      -INT32_C(          10),
      UINT32_C(   3561537) },
    { UINT32_C( 336076719),
      -INT32_C(           9),
      UINT32_C(    656400) },
    { UINT32_C(3623751117),
      -INT32_C(           9),
      UINT32_C(   7077639) },
    { UINT32_C(2268240050),
      -INT32_C(           8),
      UINT32_C(   8860313) },
    { UINT32_C(2049890110),
      -INT32_C(           8),
      UINT32_C(   8007383) },
    { UINT32_C(3255281663),
      -INT32_C(           7),
      UINT32_C(  25431888) },
    { UINT32_C(3343080295),
      -INT32_C(           7),
      UINT32_C(  26117815) },
    { UINT32_C(1371289525),
      -INT32_C(           6),
      UINT32_C(  21426399) },
    { UINT32_C(3911851322),
      -INT32_C(           6),
      UINT32_C(  61122677) },
    { UINT32_C( 167653948),
      -INT32_C(           5),
      UINT32_C(   5239186) },
    { UINT32_C(4058119231),
      -INT32_C(           5),
      UINT32_C( 126816226) },
    { UINT32_C(3531084692),
      -INT32_C(           4),
      UINT32_C( 220692793) },
    { UINT32_C(3914180586),
      -INT32_C(           4),
      UINT32_C( 244636287) },
    { UINT32_C(3064681550),
      -INT32_C(           3),
      UINT32_C( 383085194) },
    { UINT32_C(1702752175),
      -INT32_C(           3),
      UINT32_C( 212844022) },
    { UINT32_C(1622554918),
      -INT32_C(           2),
      UINT32_C( 405638730) },
    { UINT32_C(2471092311),
      -INT32_C(           2),
      UINT32_C( 617773078) },
    { UINT32_C(1386039058),
      -INT32_C(           1),
      UINT32_C( 693019529) },
    { UINT32_C(3628301891),
      -INT32_C(           1),
      UINT32_C(1814150946) },
    { UINT32_C(2091564177),
       INT32_C(           0),
      UINT32_C(2091564177) },
    { UINT32_C(2976249699),
       INT32_C(           0),
      UINT32_C(2976249699) },
    { UINT32_C(4217835851),
       INT32_C(           0),
      UINT32_C(4217835851) },
    { UINT32_C( 643884288),
       INT32_C(           1),
      UINT32_C(1287768576) },
    { UINT32_C(1971721758),
       INT32_C(           1),
      UINT32_C(3943443516) },
    { UINT32_C( 134860790),
       INT32_C(           2),
      UINT32_C( 539443160) },
    { UINT32_C(1515890199),
       INT32_C(           2),
                UINT32_MAX },
    { UINT32_C(3039993380),
       INT32_C(           3),
                UINT32_MAX },
    { UINT32_C(3174161754),
       INT32_C(           3),
                UINT32_MAX },
    { UINT32_C( 544118740),
       INT32_C(           4),
                UINT32_MAX },
    { UINT32_C(2820396712),
       INT32_C(           4),
                UINT32_MAX },
    { UINT32_C(3654187963),
       INT32_C(           5),
                UINT32_MAX },
    { UINT32_C(2270123153),
       INT32_C(           5),
                UINT32_MAX },
    { UINT32_C( 982472739),
       INT32_C(           6),
                UINT32_MAX },
    { UINT32_C( 580250366),
       INT32_C(           6),
                UINT32_MAX },
    { UINT32_C(3805792136),
       INT32_C(           7),
                UINT32_MAX },
    { UINT32_C(2040465828),
       INT32_C(           7),
                UINT32_MAX },
    { UINT32_C(1217990048),
       INT32_C(           8),
                UINT32_MAX },
    { UINT32_C(2666575075),
       INT32_C(           8),
                UINT32_MAX },
    { UINT32_C(3229138479),
       INT32_C(           9),
                UINT32_MAX },
    { UINT32_C( 893896466),
       INT32_C(           9),
                UINT32_MAX },
    { UINT32_C( 493934111),
       INT32_C(          10),
                UINT32_MAX },
    { UINT32_C(1212089792),
       INT32_C(          10),
                UINT32_MAX },
    { UINT32_C(1898583756),
       INT32_C(          11),
                UINT32_MAX },
    { UINT32_C(3219835167),
       INT32_C(          11),
                UINT32_MAX },
    { UINT32_C(3121054679),
       INT32_C(          12),
                UINT32_MAX },
    { UINT32_C(1717172023),
       INT32_C(          12),
                UINT32_MAX },
    { UINT32_C(3341210037),
       INT32_C(          13),
                UINT32_MAX },
    { UINT32_C(3086839192),
       INT32_C(          13),
                UINT32_MAX },
    { UINT32_C(  81030467),
       INT32_C(          14),
                UINT32_MAX },
    { UINT32_C(1045173106),
       INT32_C(          14),
                UINT32_MAX },
    { UINT32_C(1236236073),
       INT32_C(          15),
                UINT32_MAX },
    { UINT32_C( 386439488),
       INT32_C(          15),
                UINT32_MAX },
    { UINT32_C(1406275612),
       INT32_C(          16),
                UINT32_MAX },
    { UINT32_C(3183028999),
       INT32_C(          16),
                UINT32_MAX },
    { UINT32_C(2491736060),
       INT32_C(          17),
                UINT32_MAX },
    { UINT32_C(2420867404),
       INT32_C(          17),
                UINT32_MAX },
    { UINT32_C(1620320238),
       INT32_C(          18),
                UINT32_MAX },
    { UINT32_C(1537204274),
       INT32_C(          18),
                UINT32_MAX },
    { UINT32_C(2560904791),
       INT32_C(          19),
                UINT32_MAX },
    { UINT32_C(  78622184),
       INT32_C(          19),
                UINT32_MAX },
    { UINT32_C(3294134717),
       INT32_C(          20),
                UINT32_MAX },
    { UINT32_C(2827030956),
       INT32_C(          20),
                UINT32_MAX },
    { UINT32_C(1027344113),
       INT32_C(          21),
                UINT32_MAX },
    { UINT32_C(1993181063),
       INT32_C(          21),
                UINT32_MAX },
    { UINT32_C(3637928358),
       INT32_C(          22),
                UINT32_MAX },
    { UINT32_C(2570351938),
       INT32_C(          22),
                UINT32_MAX },
    { UINT32_C(2888947908),
       INT32_C(          23),
                UINT32_MAX },
    { UINT32_C(1118888325),
       INT32_C(          23),
                UINT32_MAX },
    { UINT32_C( 252119138),
       INT32_C(          24),
                UINT32_MAX },
    { UINT32_C( 196577306),
       INT32_C(          24),
                UINT32_MAX },
    { UINT32_C( 373879950),
       INT32_C(          25),
                UINT32_MAX },
    { UINT32_C( 579606139),
       INT32_C(          25),
                UINT32_MAX },
    { UINT32_C(3120194167),
       INT32_C(          26),
                UINT32_MAX },
    { UINT32_C(2622697176),
       INT32_C(          26),
                UINT32_MAX },
    { UINT32_C(2353562631),
       INT32_C(          27),
                UINT32_MAX },
    { UINT32_C(3369072741),
       INT32_C(          27),
                UINT32_MAX },
    { UINT32_C( 467129857),
       INT32_C(          28),
                UINT32_MAX },
    { UINT32_C(3978727006),
       INT32_C(          28),
                UINT32_MAX },
    { UINT32_C(4261637762),
       INT32_C(          29),
                UINT32_MAX },
    { UINT32_C(4229992324),
       INT32_C(          29),
                UINT32_MAX },
    { UINT32_C(3384089329),
       INT32_C(          30),
                UINT32_MAX },
    { UINT32_C(1348798537),
       INT32_C(          30),
                UINT32_MAX },
    { UINT32_C(4074548621),
       INT32_C(          31),
                UINT32_MAX },
    { UINT32_C(2814028710),
       INT32_C(          31),
                UINT32_MAX },
    { UINT32_C(3770847617),
       INT32_C(          32),
                UINT32_MAX },
    { UINT32_C(2731403296),
       INT32_C(          32),
                UINT32_MAX },
    { UINT32_C(3684749398),
       INT32_C(          33),
                UINT32_MAX },
    { UINT32_C(1356316767),
       INT32_C(          33),
                UINT32_MAX },
    { UINT32_C( 605719771),
       INT32_C(          34),
                UINT32_MAX },
    { UINT32_C( 578060181),
       INT32_C(          34),
                UINT32_MAX },
    { UINT32_C(3541323821),
       INT32_C(          35),
                UINT32_MAX },
    { UINT32_C(2105200636),
       INT32_C(          35),
                UINT32_MAX },
    { UINT32_C(2153593952),
       INT32_C(          36),
                UINT32_MAX },
    { UINT32_C(2049124900),
       INT32_C(          36),
                UINT32_MAX },
    { UINT32_C(1498792954),
       INT32_C(          37),
                UINT32_MAX },
    { UINT32_C(1604988036),
       INT32_C(          37),
                UINT32_MAX },
    { UINT32_C(1317258425),
       INT32_C(          38),
                UINT32_MAX },
    { UINT32_C(1886451523),
       INT32_C(          38),
                UINT32_MAX },
    { UINT32_C(1128498247),
       INT32_C(          39),
                UINT32_MAX },
    { UINT32_C(3032595795),
       INT32_C(          39),
                UINT32_MAX },
    { UINT32_C(         1),
      -INT32_C(          40),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          39),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          38),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          37),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          36),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          35),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          34),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          33),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          32),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          31),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          30),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          29),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          28),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          27),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          26),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          25),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          24),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          23),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          22),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          21),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          20),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          19),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          18),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          17),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          16),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          15),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          14),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          13),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          12),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          11),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(          10),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           9),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           8),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           7),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           6),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           5),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           4),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           3),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           2),
      UINT32_C(         0) },
    { UINT32_C(         1),
      -INT32_C(           1),
      UINT32_C(         1) },
    { UINT32_C(         1),
       INT32_C(           0),
      UINT32_C(         1) },
    { UINT32_C(         1),
       INT32_C(           1),
      UINT32_C(         2) },
    { UINT32_C(         1),
       INT32_C(           2),
      UINT32_C(         4) },
    { UINT32_C(         1),
       INT32_C(           3),
      UINT32_C(         8) },
    { UINT32_C(         1),
       INT32_C(           4),
      UINT32_C(        16) },
    { UINT32_C(         1),
       INT32_C(           5),
      UINT32_C(        32) },
    { UINT32_C(         1),
       INT32_C(           6),
      UINT32_C(        64) },
    { UINT32_C(         1),
       INT32_C(           7),
      UINT32_C(       128) },
    { UINT32_C(         1),
       INT32_C(           8),
      UINT32_C(       256) },
    { UINT32_C(         1),
       INT32_C(           9),
      UINT32_C(       512) },
    { UINT32_C(         1),
       INT32_C(          10),
      UINT32_C(      1024) },
    { UINT32_C(         1),
       INT32_C(          11),
      UINT32_C(      2048) },
    { UINT32_C(         1),
       INT32_C(          12),
      UINT32_C(      4096) },
    { UINT32_C(         1),
       INT32_C(          13),
      UINT32_C(      8192) },
    { UINT32_C(         1),
       INT32_C(          14),
      UINT32_C(     16384) },
    { UINT32_C(         1),
       INT32_C(          15),
      UINT32_C(     32768) },
    { UINT32_C(         1),
       INT32_C(          16),
      UINT32_C(     65536) },
    { UINT32_C(         1),
       INT32_C(          17),
      UINT32_C(    131072) },
    { UINT32_C(         1),
       INT32_C(          18),
      UINT32_C(    262144) },
    { UINT32_C(         1),
       INT32_C(          19),
      UINT32_C(    524288) },
    { UINT32_C(         1),
       INT32_C(          20),
      UINT32_C(   1048576) },
    { UINT32_C(         1),
       INT32_C(          21),
      UINT32_C(   2097152) },
    { UINT32_C(         1),
       INT32_C(          22),
      UINT32_C(   4194304) },
    { UINT32_C(         1),
       INT32_C(          23),
      UINT32_C(   8388608) },
    { UINT32_C(         1),
       INT32_C(          24),
      UINT32_C(  16777216) },
    { UINT32_C(         1),
       INT32_C(          25),
      UINT32_C(  33554432) },
    { UINT32_C(         1),
       INT32_C(          26),
      UINT32_C(  67108864) },
    { UINT32_C(         1),
       INT32_C(          27),
      UINT32_C( 134217728) },
    { UINT32_C(         1),
       INT32_C(          28),
      UINT32_C( 268435456) },
    { UINT32_C(         1),
       INT32_C(          29),
      UINT32_C( 536870912) },
    { UINT32_C(         1),
       INT32_C(          30),
      UINT32_C(1073741824) },
    { UINT32_C(         1),
       INT32_C(          31),
      UINT32_C(2147483648) },
    { UINT32_C(         1),
       INT32_C(          32),
                UINT32_MAX },
    { UINT32_C(         1),
       INT32_C(          33),
                UINT32_MAX },
    { UINT32_C(         1),
       INT32_C(          34),
                UINT32_MAX },
    { UINT32_C(         1),
       INT32_C(          35),
                UINT32_MAX },
    { UINT32_C(         1),
       INT32_C(          36),
                UINT32_MAX },
    { UINT32_C(         1),
       INT32_C(          37),
                UINT32_MAX },
    { UINT32_C(         1),
       INT32_C(          38),
                UINT32_MAX },
    { UINT32_C(         1),
       INT32_C(          39),
                UINT32_MAX },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    uint32_t r = simde_vqrshls_u32(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_u32(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    uint32_t a = simde_test_codegen_random_u32();
    int32_t b = simde_test_codegen_random_i32();
    uint32_t r = simde_vqrshls_u32(a, b);

    simde_test_codegen_write_u32(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -80 ; i < 80 ; i++) {
    uint32_t a = simde_test_codegen_random_u32();
    int32_t b = i / 2;
    uint32_t r = simde_vqrshls_u32(a, b);

    simde_test_codegen_write_u32(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -40 ; i < 40 ; i++) {
    uint32_t a = 1;
    int32_t b = i;
    uint32_t r = simde_vqrshls_u32(a, b);

    simde_test_codegen_write_u32(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshld_u64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    uint64_t a;
    int64_t b;
    uint64_t r;
  } test_vec[] = {
    { UINT64_C( 7429138017364722758),
       INT64_C( 4619745276961468766),
                          UINT64_MAX },
    { UINT64_C(16729366776128966668),
      -INT64_C( 6295118163883541953),
                          UINT64_MAX },
    { UINT64_C(18020346440811283203),
      -INT64_C( 2959584269102208813),
      UINT64_C(              512169) },
    { UINT64_C( 3917121183626973185),
       INT64_C( 4264310951252916258),
                          UINT64_MAX },
    { UINT64_C( 3061038548513273571),
       INT64_C( 4739692536268312102),
                          UINT64_MAX },
    { UINT64_C(10322832451653927957),
       INT64_C( 4718436907658542601),
                          UINT64_MAX },
    { UINT64_C( 9942042947975472247),
      -INT64_C( 6085982031727439607),
                          UINT64_MAX },
    { UINT64_C( 6430851442872368963),
       INT64_C(  799412662857978488),
                          UINT64_MAX },
    { UINT64_C( 2118332385817450873),
      -INT64_C(                  70),
      UINT64_C(                   0) },
    { UINT64_C( 6982532280625847973),
      -INT64_C(                  69),
      UINT64_C(                   0) },
    { UINT64_C( 1141748918117178927),
      -INT64_C(                  68),
      UINT64_C(                   0) },
    { UINT64_C(   18294255171039904),
      -INT64_C(                  67),
      UINT64_C(                   0) },
    { UINT64_C(10594790617804184279),
      -INT64_C(                  66),
      UINT64_C(                   0) },
    { UINT64_C(10348345464437239583),
      -INT64_C(                  65),
      UINT64_C(                   0) },
    { UINT64_C(11167493290873236563),
      -INT64_C(                  64),
      UINT64_C(                   1) },
    { UINT64_C(13105492365577019481),
      -INT64_C(                  63),
      UINT64_C(                   1) },
    { UINT64_C( 2210045851443707177),
      -INT64_C(                  62),
      UINT64_C(                   0) },
    { UINT64_C( 9369447858028837399),
      -INT64_C(                  61),
      UINT64_C(                   4) },
    { UINT64_C( 6068432253768963235),
      -INT64_C(                  60),
      UINT64_C(                   5) },
    { UINT64_C( 4600731674881558086),
      -INT64_C(                  59),
      UINT64_C(                   8) },
    { UINT64_C( 4174276253166565842),
      -INT64_C(                  58),
      UINT64_C(                  14) },
    { UINT64_C(16369940609436240252),
      -INT64_C(                  57),
      UINT64_C(                 114) },
    { UINT64_C( 5123890215022271179),
      -INT64_C(                  56),
      UINT64_C(                  71) },
    { UINT64_C( 3896252926974971838),
      -INT64_C(                  55),
      UINT64_C(                 108) },
    { UINT64_C(16724950014157613502),
      -INT64_C(                  54),
      UINT64_C(                 928) },
    { UINT64_C( 5503892981542079038),
      -INT64_C(                  53),
      UINT64_C(                 611) },
    { UINT64_C( 6314941239201554779),
      -INT64_C(                  52),
      UINT64_C(                1402) },
    { UINT64_C( 1303859522592968317),
      -INT64_C(                  51),
      UINT64_C(                 579) },
    { UINT64_C( 6051443804659130278),
      -INT64_C(                  50),
      UINT64_C(                5375) },
    { UINT64_C(12440124490884681687),
      -INT64_C(                  49),
      UINT64_C(               22098) },
    { UINT64_C( 6990901520229774868),
      -INT64_C(                  48),
      UINT64_C(               24837) },
    { UINT64_C( 1305861837086142393),
      -INT64_C(                  47),
      UINT64_C(                9279) },
    { UINT64_C(17717477724658143609),
      -INT64_C(                  46),
      UINT64_C(              251781) },
    { UINT64_C(18062796700632593071),
      -INT64_C(                  45),
      UINT64_C(              513376) },
    { UINT64_C( 1795999873483912103),
      -INT64_C(                  44),
      UINT64_C(              102091) },
    { UINT64_C(  241299411435859931),
      -INT64_C(                  43),
      UINT64_C(               27433) },
    { UINT64_C( 5577834163367673429),
      -INT64_C(                  42),
      UINT64_C(             1268253) },
    { UINT64_C( 5621891626886346020),
      -INT64_C(                  41),
      UINT64_C(             2556540) },
    { UINT64_C( 3612667928251050924),
      -INT64_C(                  40),
      UINT64_C(             3285702) },
    { UINT64_C(16249501526605252597),
      -INT64_C(                  39),
      UINT64_C(            29557671) },
    { UINT64_C(12195111744654545489),
      -INT64_C(                  38),
      UINT64_C(            44365558) },
    { UINT64_C( 3415900616083606130),
      -INT64_C(                  37),
      UINT64_C(            24853948) },
    { UINT64_C(18152985381987500130),
      -INT64_C(                  36),
      UINT64_C(           264160704) },
    { UINT64_C( 9948727799889684131),
      -INT64_C(                  35),
      UINT64_C(           289546087) },
    { UINT64_C(10681333395570189326),
      -INT64_C(                  34),
      UINT64_C(           621735432) },
    { UINT64_C(  645163348839493186),
      -INT64_C(                  33),
      UINT64_C(            75106899) },
    { UINT64_C( 3266990488972045908),
      -INT64_C(                  32),
      UINT64_C(           760655498) },
    { UINT64_C(15769473719964848148),
      -INT64_C(                  31),
      UINT64_C(          7343233433) },
    { UINT64_C( 5586337152770003665),
      -INT64_C(                  30),
      UINT64_C(          5202681900) },
    { UINT64_C(11260132970864980886),
      -INT64_C(                  29),
      UINT64_C(         20973632058) },
    { UINT64_C(14550089934837319221),
      -INT64_C(                  28),
      UINT64_C(         54203308876) },
    { UINT64_C( 9711149161601452318),
      -INT64_C(                  27),
      UINT64_C(         72353699517) },
    { UINT64_C( 8041458439114272833),
      -INT64_C(                  26),
      UINT64_C(        119827068435) },
    { UINT64_C(16651760222090275635),
      -INT64_C(                  25),
      UINT64_C(        496261126461) },
    { UINT64_C( 8751217948320206325),
      -INT64_C(                  24),
      UINT64_C(        521613237162) },
    { UINT64_C(13265114621936896475),
      -INT64_C(                  23),
      UINT64_C(       1581324889891) },
    { UINT64_C(14096157501787770967),
      -INT64_C(                  22),
      UINT64_C(       3360785842368) },
    { UINT64_C( 3377792328499284551),
      -INT64_C(                  21),
      UINT64_C(       1610656894922) },
    { UINT64_C(17597052593422960225),
      -INT64_C(                  20),
      UINT64_C(      16781857102797) },
    { UINT64_C( 1658342532717058259),
      -INT64_C(                  19),
      UINT64_C(       3163037362513) },
    { UINT64_C(16969816986861635373),
      -INT64_C(                  18),
      UINT64_C(      64734714457938) },
    { UINT64_C( 8329342718721703472),
      -INT64_C(                  17),
      UINT64_C(      63547841787122) },
    { UINT64_C(10041168686828514891),
      -INT64_C(                  16),
      UINT64_C(     153216074933296) },
    { UINT64_C(12237645212961660670),
      -INT64_C(                  15),
      UINT64_C(     373463293852590) },
    { UINT64_C( 3562038348982131680),
      -INT64_C(                  14),
      UINT64_C(     217409567198616) },
    { UINT64_C( 7145577741006033829),
      -INT64_C(                  13),
      UINT64_C(     872262907837651) },
    { UINT64_C(13755156271903300256),
      -INT64_C(                  12),
      UINT64_C(    3358192449195142) },
    { UINT64_C( 8296845007885417215),
      -INT64_C(                  11),
      UINT64_C(    4051193851506551) },
    { UINT64_C( 1629851653005281899),
      -INT64_C(                  10),
      UINT64_C(    1591652004887971) },
    { UINT64_C(  735204301844855224),
      -INT64_C(                   9),
      UINT64_C(    1435945902040733) },
    { UINT64_C( 2076237614110997330),
      -INT64_C(                   8),
      UINT64_C(    8110303180121083) },
    { UINT64_C(  506579764284705793),
      -INT64_C(                   7),
      UINT64_C(    3957654408474264) },
    { UINT64_C( 1931221129917431109),
      -INT64_C(                   6),
      UINT64_C(   30175330154959861) },
    { UINT64_C(16555547324968508236),
      -INT64_C(                   5),
      UINT64_C(  517360853905265882) },
    { UINT64_C( 7805521466995778585),
      -INT64_C(                   4),
      UINT64_C(  487845091687236162) },
    { UINT64_C(16841219210460850452),
      -INT64_C(                   3),
      UINT64_C( 2105152401307606307) },
    { UINT64_C(10052437061868203012),
      -INT64_C(                   2),
      UINT64_C( 2513109265467050753) },
    { UINT64_C(11781701994039897097),
      -INT64_C(                   1),
      UINT64_C( 5890850997019948549) },
    { UINT64_C( 6268495296277870887),
       INT64_C(                   0),
      UINT64_C( 6268495296277870887) },
    { UINT64_C(  694327323851750932),
       INT64_C(                   1),
      UINT64_C( 1388654647703501864) },
    { UINT64_C(13382150181234950891),
       INT64_C(                   2),
                          UINT64_MAX },
    { UINT64_C( 2489320695987042398),
       INT64_C(                   3),
                          UINT64_MAX },
    { UINT64_C(11955283019838022992),
       INT64_C(                   4),
                          UINT64_MAX },
    { UINT64_C(11150912332519829680),
       INT64_C(                   5),
                          UINT64_MAX },
    { UINT64_C( 9199864624059854658),
       INT64_C(                   6),
                          UINT64_MAX },
    { UINT64_C( 2995724813079085428),
       INT64_C(                   7),
                          UINT64_MAX },
    { UINT64_C(12289660351422351192),
       INT64_C(                   8),
                          UINT64_MAX },
    { UINT64_C( 3233055574000135060),
       INT64_C(                   9),
                          UINT64_MAX },
    { UINT64_C(10732019772915950414),
       INT64_C(                  10),
                          UINT64_MAX },
    { UINT64_C(11659421728354096844),
       INT64_C(                  11),
                          UINT64_MAX },
    { UINT64_C(14015509005305809373),
       INT64_C(                  12),
                          UINT64_MAX },
    { UINT64_C( 5555723403182661634),
       INT64_C(                  13),
                          UINT64_MAX },
    { UINT64_C(15004774615702321287),
       INT64_C(                  14),
                          UINT64_MAX },
    { UINT64_C(12965093485642596098),
       INT64_C(                  15),
                          UINT64_MAX },
    { UINT64_C(10501562614814504908),
       INT64_C(                  16),
                          UINT64_MAX },
    { UINT64_C( 3330297196913279979),
       INT64_C(                  17),
                          UINT64_MAX },
    { UINT64_C(11647171339950093520),
       INT64_C(                  18),
                          UINT64_MAX },
    { UINT64_C( 2943342441499793487),
       INT64_C(                  19),
                          UINT64_MAX },
    { UINT64_C( 6372839135717253713),
       INT64_C(                  20),
                          UINT64_MAX },
    { UINT64_C( 6230864801640476322),
       INT64_C(                  21),
                          UINT64_MAX },
    { UINT64_C(12515447693334105454),
       INT64_C(                  22),
                          UINT64_MAX },
    { UINT64_C(14369203510989462757),
       INT64_C(                  23),
                          UINT64_MAX },
    { UINT64_C(  995884217972263692),
       INT64_C(                  24),
                          UINT64_MAX },
    { UINT64_C(18366470445812094214),
       INT64_C(                  25),
                          UINT64_MAX },
    { UINT64_C(13702408404709673756),
       INT64_C(                  26),
                          UINT64_MAX },
    { UINT64_C(12796530452571617880),
       INT64_C(                  27),
                          UINT64_MAX },
    { UINT64_C( 6515861299697237491),
       INT64_C(                  28),
                          UINT64_MAX },
    { UINT64_C(12582996416003490578),
       INT64_C(                  29),
                          UINT64_MAX },
    { UINT64_C(13792926314487458772),
       INT64_C(                  30),
                          UINT64_MAX },
    { UINT64_C(14240984356289866059),
       INT64_C(                  31),
                          UINT64_MAX },
    { UINT64_C(15857050913779993952),
       INT64_C(                  32),
                          UINT64_MAX },
    { UINT64_C(10561818256828397138),
       INT64_C(                  33),
                          UINT64_MAX },
    { UINT64_C( 6408245193887169736),
       INT64_C(                  34),
                          UINT64_MAX },
    { UINT64_C(   47828343026449427),
       INT64_C(                  35),
                          UINT64_MAX },
    { UINT64_C(12636753506057178552),
       INT64_C(                  36),
                          UINT64_MAX },
    { UINT64_C(13910254722697386938),
       INT64_C(                  37),
                          UINT64_MAX },
    { UINT64_C(  524915599679179490),
       INT64_C(                  38),
                          UINT64_MAX },
    { UINT64_C( 3817527998168142171),
       INT64_C(                  39),
                          UINT64_MAX },
    { UINT64_C( 3345352589854551906),
       INT64_C(                  40),
                          UINT64_MAX },
    { UINT64_C(  768342195431735791),
       INT64_C(                  41),
                          UINT64_MAX },
    { UINT64_C(15274030123938585085),
       INT64_C(                  42),
                          UINT64_MAX },
    { UINT64_C(15817425264617383122),
       INT64_C(                  43),
                          UINT64_MAX },
    { UINT64_C( 4330782018638526584),
       INT64_C(                  44),
                          UINT64_MAX },
    { UINT64_C( 8230835697182097928),
       INT64_C(                  45),
                          UINT64_MAX },
    { UINT64_C( 5166293379581227878),
       INT64_C(                  46),
                          UINT64_MAX },
    { UINT64_C(13948800650798530624),
       INT64_C(                  47),
                          UINT64_MAX },
    { UINT64_C( 4456812339352383056),
       INT64_C(                  48),
                          UINT64_MAX },
    { UINT64_C( 3490754576971005321),
       INT64_C(                  49),
                          UINT64_MAX },
    { UINT64_C( 8256437533879031257),
       INT64_C(                  50),
                          UINT64_MAX },
    { UINT64_C(10542012219802724930),
       INT64_C(                  51),
                          UINT64_MAX },
    { UINT64_C( 5359293182182947310),
       INT64_C(                  52),
                          UINT64_MAX },
    { UINT64_C( 5404524555111353349),
       INT64_C(                  53),
                          UINT64_MAX },
    { UINT64_C( 1455276008496762767),
       INT64_C(                  54),
                          UINT64_MAX },
    { UINT64_C(  877335643600936535),
       INT64_C(                  55),
                          UINT64_MAX },
    { UINT64_C( 7634040505817324517),
       INT64_C(                  56),
                          UINT64_MAX },
    { UINT64_C( 9446645416980986459),
       INT64_C(                  57),
                          UINT64_MAX },
    { UINT64_C( 3976246225900122696),
       INT64_C(                  58),
                          UINT64_MAX },
    { UINT64_C(11528931071199661524),
       INT64_C(                  59),
                          UINT64_MAX },
    { UINT64_C( 1668080896227749208),
       INT64_C(                  60),
                          UINT64_MAX },
    { UINT64_C(16206202814108884848),
       INT64_C(                  61),
                          UINT64_MAX },
    { UINT64_C( 8487359647160347569),
       INT64_C(                  62),
                          UINT64_MAX },
    { UINT64_C(17243831725405477622),
       INT64_C(                  63),
                          UINT64_MAX },
    { UINT64_C( 4280058787230684306),
       INT64_C(                  64),
                          UINT64_MAX },
    { UINT64_C(18134174129864002353),
       INT64_C(                  65),
                          UINT64_MAX },
    { UINT64_C(14035563674275259570),
       INT64_C(                  66),
                          UINT64_MAX },
    { UINT64_C( 2771614364853219312),
       INT64_C(                  67),
                          UINT64_MAX },
    { UINT64_C( 2691703369830244948),
       INT64_C(                  68),
                          UINT64_MAX },
    { UINT64_C( 3632883726812117284),
       INT64_C(                  69),
                          UINT64_MAX },
    { UINT64_C(                   1),
      -INT64_C(                  70),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  69),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  68),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  67),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  66),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  65),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  64),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  63),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  62),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  61),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  60),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  59),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  58),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  57),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  56),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  55),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  54),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  53),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  52),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  51),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  50),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  49),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  48),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  47),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  46),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  45),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  44),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  43),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  42),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  41),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  40),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  39),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  38),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  37),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  36),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  35),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  34),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  33),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  32),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  31),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  30),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  29),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  28),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  27),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  26),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  25),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  24),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  23),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  22),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  21),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  20),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  19),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  18),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  17),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  16),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  15),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  14),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  13),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  12),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  11),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                  10),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   9),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   8),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   7),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   6),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   5),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   4),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   3),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   2),
      UINT64_C(                   0) },
    { UINT64_C(                   1),
      -INT64_C(                   1),
      UINT64_C(                   1) },
    { UINT64_C(                   1),
       INT64_C(                   0),
      UINT64_C(                   1) },
    { UINT64_C(                   1),
       INT64_C(                   1),
      UINT64_C(                   2) },
    { UINT64_C(                   1),
       INT64_C(                   2),
      UINT64_C(                   4) },
    { UINT64_C(                   1),
       INT64_C(                   3),
      UINT64_C(                   8) },
    { UINT64_C(                   1),
       INT64_C(                   4),
      UINT64_C(                  16) },
    { UINT64_C(                   1),
       INT64_C(                   5),
      UINT64_C(                  32) },
    { UINT64_C(                   1),
       INT64_C(                   6),
      UINT64_C(                  64) },
    { UINT64_C(                   1),
       INT64_C(                   7),
      UINT64_C(                 128) },
    { UINT64_C(                   1),
       INT64_C(                   8),
      UINT64_C(                 256) },
    { UINT64_C(                   1),
       INT64_C(                   9),
      UINT64_C(                 512) },
    { UINT64_C(                   1),
       INT64_C(                  10),
      UINT64_C(                1024) },
    { UINT64_C(                   1),
       INT64_C(                  11),
      UINT64_C(                2048) },
    { UINT64_C(                   1),
       INT64_C(                  12),
      UINT64_C(                4096) },
    { UINT64_C(                   1),
       INT64_C(                  13),
      UINT64_C(                8192) },
    { UINT64_C(                   1),
       INT64_C(                  14),
      UINT64_C(               16384) },
    { UINT64_C(                   1),
       INT64_C(                  15),
      UINT64_C(               32768) },
    { UINT64_C(                   1),
       INT64_C(                  16),
      UINT64_C(               65536) },
    { UINT64_C(                   1),
       INT64_C(                  17),
      UINT64_C(              131072) },
    { UINT64_C(                   1),
       INT64_C(                  18),
      UINT64_C(              262144) },
    { UINT64_C(                   1),
       INT64_C(                  19),
      UINT64_C(              524288) },
    { UINT64_C(                   1),
       INT64_C(                  20),
      UINT64_C(             1048576) },
    { UINT64_C(                   1),
       INT64_C(                  21),
      UINT64_C(             2097152) },
    { UINT64_C(                   1),
       INT64_C(                  22),
      UINT64_C(             4194304) },
    { UINT64_C(                   1),
       INT64_C(                  23),
      UINT64_C(             8388608) },
    { UINT64_C(                   1),
       INT64_C(                  24),
      UINT64_C(            16777216) },
    { UINT64_C(                   1),
       INT64_C(                  25),
      UINT64_C(            33554432) },
    { UINT64_C(                   1),
       INT64_C(                  26),
      UINT64_C(            67108864) },
    { UINT64_C(                   1),
       INT64_C(                  27),
      UINT64_C(           134217728) },
    { UINT64_C(                   1),
       INT64_C(                  28),
      UINT64_C(           268435456) },
    { UINT64_C(                   1),
       INT64_C(                  29),
      UINT64_C(           536870912) },
    { UINT64_C(                   1),
       INT64_C(                  30),
      UINT64_C(          1073741824) },
    { UINT64_C(                   1),
       INT64_C(                  31),
      UINT64_C(          2147483648) },
    { UINT64_C(                   1),
       INT64_C(                  32),
      UINT64_C(          4294967296) },
    { UINT64_C(                   1),
       INT64_C(                  33),
      UINT64_C(          8589934592) },
    { UINT64_C(                   1),
       INT64_C(                  34),
      UINT64_C(         17179869184) },
    { UINT64_C(                   1),
       INT64_C(                  35),
      UINT64_C(         34359738368) },
    { UINT64_C(                   1),
       INT64_C(                  36),
      UINT64_C(         68719476736) },
    { UINT64_C(                   1),
       INT64_C(                  37),
      UINT64_C(        137438953472) },
    { UINT64_C(                   1),
       INT64_C(                  38),
      UINT64_C(        274877906944) },
    { UINT64_C(                   1),
       INT64_C(                  39),
      UINT64_C(        549755813888) },
    { UINT64_C(                   1),
       INT64_C(                  40),
      UINT64_C(       1099511627776) },
    { UINT64_C(                   1),
       INT64_C(                  41),
      UINT64_C(       2199023255552) },
    { UINT64_C(                   1),
       INT64_C(                  42),
      UINT64_C(       4398046511104) },
    { UINT64_C(                   1),
       INT64_C(                  43),
      UINT64_C(       8796093022208) },
    { UINT64_C(                   1),
       INT64_C(                  44),
      UINT64_C(      17592186044416) },
    { UINT64_C(                   1),
       INT64_C(                  45),
      UINT64_C(      35184372088832) },
    { UINT64_C(                   1),
       INT64_C(                  46),
      UINT64_C(      70368744177664) },
    { UINT64_C(                   1),
       INT64_C(                  47),
      UINT64_C(     140737488355328) },
    { UINT64_C(                   1),
       INT64_C(                  48),
      UINT64_C(     281474976710656) },
    { UINT64_C(                   1),
       INT64_C(                  49),
      UINT64_C(     562949953421312) },
    { UINT64_C(                   1),
       INT64_C(                  50),
      UINT64_C(    1125899906842624) },
    { UINT64_C(                   1),
       INT64_C(                  51),
      UINT64_C(    2251799813685248) },
    { UINT64_C(                   1),
       INT64_C(                  52),
      UINT64_C(    4503599627370496) },
    { UINT64_C(                   1),
       INT64_C(                  53),
      UINT64_C(    9007199254740992) },
    { UINT64_C(                   1),
       INT64_C(                  54),
      UINT64_C(   18014398509481984) },
    { UINT64_C(                   1),
       INT64_C(                  55),
      UINT64_C(   36028797018963968) },
    { UINT64_C(                   1),
       INT64_C(                  56),
      UINT64_C(   72057594037927936) },
    { UINT64_C(                   1),
       INT64_C(                  57),
      UINT64_C(  144115188075855872) },
    { UINT64_C(                   1),
       INT64_C(                  58),
      UINT64_C(  288230376151711744) },
    { UINT64_C(                   1),
       INT64_C(                  59),
      UINT64_C(  576460752303423488) },
    { UINT64_C(                   1),
       INT64_C(                  60),
      UINT64_C( 1152921504606846976) },
    { UINT64_C(                   1),
       INT64_C(                  61),
      UINT64_C( 2305843009213693952) },
    { UINT64_C(                   1),
       INT64_C(                  62),
      UINT64_C( 4611686018427387904) },
    { UINT64_C(                   1),
       INT64_C(                  63),
      UINT64_C( 9223372036854775808) },
    { UINT64_C(                   1),
       INT64_C(                  64),
                          UINT64_MAX },
    { UINT64_C(                   1),
       INT64_C(                  65),
                          UINT64_MAX },
    { UINT64_C(                   1),
       INT64_C(                  66),
                          UINT64_MAX },
    { UINT64_C(                   1),
       INT64_C(                  67),
                          UINT64_MAX },
    { UINT64_C(                   1),
       INT64_C(                  68),
                          UINT64_MAX },
    { UINT64_C(                   1),
       INT64_C(                  69),
                          UINT64_MAX },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    uint64_t r = simde_vqrshld_u64(test_vec[i].a, test_vec[i].b);

    simde_assert_equal_u64(r, test_vec[i].r);
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    uint64_t a = simde_test_codegen_random_u64();
    int64_t b = simde_test_codegen_random_i64();
    uint64_t r = simde_vqrshld_u64(a, b);

    simde_test_codegen_write_u64(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i64(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -70 ; i < 70 ; i++) {
    uint64_t a = simde_test_codegen_random_u64();
    int64_t b = i;
    uint64_t r = simde_vqrshld_u64(a, b);

    simde_test_codegen_write_u64(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i64(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  for (int i = -70 ; i < 70 ; i++) {
    uint64_t a = 1;
    int64_t b = i;
    uint64_t r = simde_vqrshld_u64(a, b);

    simde_test_codegen_write_u64(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_codegen_write_i64(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_u64(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshl_s8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int8_t a[8];
    int8_t b[8];
    int8_t r[8];
  } test_vec[] = {
    { { -INT8_C(   87), -INT8_C(    1),  INT8_C(   80),  INT8_C(   77),
         INT8_C(   56), -INT8_C(   54),  INT8_C(  102),  INT8_C(  111) },
      {  INT8_C(    8),  INT8_C(    3), -INT8_C(    7),  INT8_C(    2),
         INT8_C(    1),  INT8_C(    7),  INT8_C(    4), -INT8_C(    7) },
      {       INT8_MIN, -INT8_C(    8),  INT8_C(    1),       INT8_MAX,
         INT8_C(  112),       INT8_MIN,       INT8_MAX,  INT8_C(    1) } },
    { {  INT8_C(   42),  INT8_C(   74), -INT8_C(  106), -INT8_C(   29),
        -INT8_C(   94), -INT8_C(   66), -INT8_C(   22),  INT8_C(   88) },
      {  INT8_C(    5),  INT8_C(    4), -INT8_C(    5),  INT8_C(    1),
        -INT8_C(    7), -INT8_C(    6), -INT8_C(    4),  INT8_C(    0) },
      {       INT8_MAX,       INT8_MAX, -INT8_C(    3), -INT8_C(   58),
        -INT8_C(    1), -INT8_C(    1), -INT8_C(    1),  INT8_C(   88) } },
    { {       INT8_MIN,  INT8_C(  100),  INT8_C(    4),  INT8_C(   14),
         INT8_C(   92),  INT8_C(   74),  INT8_C(   59), -INT8_C(   98) },
      { -INT8_C(    3),  INT8_C(    8),  INT8_C(    8),  INT8_C(    6),
         INT8_C(    5),  INT8_C(    5),  INT8_C(    5), -INT8_C(    6) },
      { -INT8_C(   16),       INT8_MAX,       INT8_MAX,       INT8_MAX,
              INT8_MAX,       INT8_MAX,       INT8_MAX, -INT8_C(    2) } },
    { {  INT8_C(   40),  INT8_C(   61),  INT8_C(   73), -INT8_C(   94),
         INT8_C(  120),  INT8_C(   65),  INT8_C(   61), -INT8_C(   71) },
      {  INT8_C(    7),  INT8_C(    1), -INT8_C(    6), -INT8_C(    8),
         INT8_C(    6), -INT8_C(    5), -INT8_C(    4), -INT8_C(    6) },
      {       INT8_MAX,  INT8_C(  122),  INT8_C(    1),  INT8_C(    0),
              INT8_MAX,  INT8_C(    2),  INT8_C(    4), -INT8_C(    1) } },
    { {  INT8_C(   32), -INT8_C(   33), -INT8_C(   42), -INT8_C(  124),
         INT8_C(   13), -INT8_C(   93),  INT8_C(   29),  INT8_C(   93) },
      { -INT8_C(    8), -INT8_C(    1),  INT8_C(    1), -INT8_C(    1),
         INT8_C(    7), -INT8_C(    2),  INT8_C(    0),  INT8_C(    6) },
      {  INT8_C(    0), -INT8_C(   16), -INT8_C(   84), -INT8_C(   62),
              INT8_MAX, -INT8_C(   23),  INT8_C(   29),       INT8_MAX } },
    { {  INT8_C(  109), -INT8_C(  121), -INT8_C(   67), -INT8_C(   26),
         INT8_C(  116), -INT8_C(   14), -INT8_C(  107),  INT8_C(  120) },
      { -INT8_C(    8),  INT8_C(    1), -INT8_C(    7),  INT8_C(    2),
         INT8_C(    1), -INT8_C(    5),  INT8_C(    2),  INT8_C(    2) },
      {  INT8_C(    0),       INT8_MIN, -INT8_C(    1), -INT8_C(  104),
              INT8_MAX,  INT8_C(    0),       INT8_MIN,       INT8_MAX } },
    { { -INT8_C(   75),  INT8_C(    9),  INT8_C(   17),  INT8_C(    9),
        -INT8_C(   82),  INT8_C(   82), -INT8_C(   91), -INT8_C(   24) },
      {  INT8_C(    1),  INT8_C(    7),  INT8_C(    4), -INT8_C(    4),
         INT8_C(    0),  INT8_C(    6), -INT8_C(    8), -INT8_C(    7) },
      {       INT8_MIN,       INT8_MAX,       INT8_MAX,  INT8_C(    1),
        -INT8_C(   82),       INT8_MAX,  INT8_C(    0),  INT8_C(    0) } },
    { { -INT8_C(    3), -INT8_C(  107),  INT8_C(   50), -INT8_C(   31),
              INT8_MIN,  INT8_C(   61),  INT8_C(   53), -INT8_C(    4) },
      { -INT8_C(    5),  INT8_C(    5),  INT8_C(    6),  INT8_C(    2),
        -INT8_C(    2),  INT8_C(    5), -INT8_C(    6), -INT8_C(    5) },
      {  INT8_C(    0),       INT8_MIN,       INT8_MAX, -INT8_C(  124),
        -INT8_C(   32),       INT8_MAX,  INT8_C(    1),  INT8_C(    0) } },
    { {       INT8_MAX,       INT8_MAX,       INT8_MAX,       INT8_MAX,
              INT8_MAX,       INT8_MAX,       INT8_MAX,       INT8_MAX },
      { -INT8_C(    1), -INT8_C(    1), -INT8_C(    1), -INT8_C(    1),
        -INT8_C(    1), -INT8_C(    1), -INT8_C(    1), -INT8_C(    1) },
      {  INT8_C(   64),  INT8_C(   64),  INT8_C(   64),  INT8_C(   64),
         INT8_C(   64),  INT8_C(   64),  INT8_C(   64),  INT8_C(   64) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int8x8_t a = simde_vld1_s8(test_vec[i].a);
    simde_int8x8_t b = simde_vld1_s8(test_vec[i].b);
    simde_int8x8_t r = simde_vqrshl_s8(a, b);

    simde_test_arm_neon_assert_equal_i8x8(r, simde_vld1_s8(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int8x8_t a = simde_test_arm_neon_random_i8x8();
    simde_int8x8_t b = simde_test_arm_neon_random_i8x8();
    simde_int8x8_t r = simde_vqrshl_s8(a, b);

    simde_test_arm_neon_write_i8x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i8x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i8x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}


static int
test_simde_vqrshl_s16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int16_t a[4];
    int16_t b[4];
    int16_t r[4];
  } test_vec[] = {
    { {  INT16_C(   31263),  INT16_C(   18737),  INT16_C(   14706), -INT16_C(   22084) },
      {  INT16_C(       0), -INT16_C(       6),  INT16_C(       5),  INT16_C(      11) },
      {  INT16_C(   31263),  INT16_C(     293),          INT16_MAX,          INT16_MIN } },
    { { -INT16_C(   27185),  INT16_C(    1410),  INT16_C(   26995), -INT16_C(    8974) },
      { -INT16_C(      15),  INT16_C(       9), -INT16_C(       4),  INT16_C(       4) },
      { -INT16_C(       1),          INT16_MAX,  INT16_C(    1687),          INT16_MIN } },
    { { -INT16_C(    5300),  INT16_C(    1064), -INT16_C(    4694),  INT16_C(   25472) },
      {  INT16_C(       3),  INT16_C(       0),  INT16_C(       5),  INT16_C(       8) },
      {          INT16_MIN,  INT16_C(    1064),          INT16_MIN,          INT16_MAX } },
    { { -INT16_C(    9080),  INT16_C(   16802), -INT16_C(   10592), -INT16_C(   30376) },
      { -INT16_C(       4), -INT16_C(       8),  INT16_C(       0),  INT16_C(      16) },
      { -INT16_C(     567),  INT16_C(      66), -INT16_C(   10592),          INT16_MIN } },
    { { -INT16_C(   10167),  INT16_C(   25018),  INT16_C(   28423), -INT16_C(    9608) },
      { -INT16_C(       2), -INT16_C(       4), -INT16_C(       6), -INT16_C(       8) },
      { -INT16_C(    2542),  INT16_C(    1564),  INT16_C(     444), -INT16_C(      38) } },
    { {  INT16_C(   32142), -INT16_C(   28370),  INT16_C(   24856),  INT16_C(   26195) },
      { -INT16_C(       3), -INT16_C(       8),  INT16_C(      15),  INT16_C(       9) },
      {  INT16_C(    4018), -INT16_C(     111),          INT16_MAX,          INT16_MAX } },
    { { -INT16_C(   23494), -INT16_C(   25691), -INT16_C(    8401), -INT16_C(   31132) },
      {  INT16_C(       3), -INT16_C(      13), -INT16_C(      16),  INT16_C(      14) },
      {          INT16_MIN, -INT16_C(       3),  INT16_C(       0),          INT16_MIN } },
    { {  INT16_C(   14453), -INT16_C(   12196),  INT16_C(   27445),  INT16_C(   31840) },
      {  INT16_C(      14),  INT16_C(       5),  INT16_C(      13), -INT16_C(       9) },
      {          INT16_MAX,          INT16_MIN,          INT16_MAX,  INT16_C(      62) } },
    { {          INT16_MAX,          INT16_MAX,          INT16_MAX,          INT16_MAX },
      { -INT16_C(       1), -INT16_C(       1), -INT16_C(       1), -INT16_C(       1) },
      {  INT16_C(   16384),  INT16_C(   16384),  INT16_C(   16384),  INT16_C(   16384) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int16x4_t a = simde_vld1_s16(test_vec[i].a);
    simde_int16x4_t b = simde_vld1_s16(test_vec[i].b);
    simde_int16x4_t r = simde_vqrshl_s16(a, b);

    simde_test_arm_neon_assert_equal_i16x4(r, simde_vld1_s16(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int16x4_t a = simde_test_arm_neon_random_i16x4();
    simde_int16x4_t b = simde_test_arm_neon_random_i16x4();
    simde_int16x4_t r = simde_vqrshl_s16(a, b);

    simde_test_arm_neon_write_i16x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i16x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i16x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshl_s32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int32_t a[2];
    int32_t b[2];
    int32_t r[2];
  } test_vec[] = {
    { {  INT32_C(    37316345),  INT32_C(   852588016) },
      { -INT32_C(          21),  INT32_C(          22) },
      {  INT32_C(          18),              INT32_MAX } },
    { {  INT32_C(  1713047364), -INT32_C(   718155773) },
      { -INT32_C(           2),  INT32_C(          18) },
      {  INT32_C(   428261841),              INT32_MIN } },
    { {  INT32_C(  2011282434), -INT32_C(  1644508231) },
      {  INT32_C(          12), -INT32_C(          21) },
      {              INT32_MAX, -INT32_C(         784) } },
    { { -INT32_C(     8819381), -INT32_C(   112184656) },
      { -INT32_C(          12), -INT32_C(          21) },
      { -INT32_C(        2153), -INT32_C(          53) } },
    { {  INT32_C(   305438996),  INT32_C(  1662430416) },
      {  INT32_C(          29), -INT32_C(          19) },
      {              INT32_MAX,  INT32_C(        3171) } },
    { {  INT32_C(  1246050521), -INT32_C(   703046011) },
      {  INT32_C(          11),  INT32_C(          11) },
      {              INT32_MAX,              INT32_MIN } },
    { { -INT32_C(   129492298), -INT32_C(  1643446120) },
      {  INT32_C(          13),  INT32_C(          25) },
      {              INT32_MIN,              INT32_MIN } },
    { {  INT32_C(    58593943), -INT32_C(   594339506) },
      { -INT32_C(          21), -INT32_C(           9) },
      {  INT32_C(          28), -INT32_C(     1160819) } },
    { {              INT32_MAX,              INT32_MAX },
      { -INT32_C(           1), -INT32_C(           1) },
      {  INT32_C(  1073741824),  INT32_C(  1073741824) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int32x2_t a = simde_vld1_s32(test_vec[i].a);
    simde_int32x2_t b = simde_vld1_s32(test_vec[i].b);
    simde_int32x2_t r = simde_vqrshl_s32(a, b);

    simde_test_arm_neon_assert_equal_i32x2(r, simde_vld1_s32(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int32x2_t a = simde_test_arm_neon_random_i32x2();
    simde_int32x2_t b = simde_test_arm_neon_random_i32x2();
    simde_int32x2_t r = simde_vqrshl_s32(a, b);

    simde_test_arm_neon_write_i32x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i32x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i32x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshl_s64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int64_t a[1];
    int64_t b[1];
    int64_t r[1];
  } test_vec[] = {
    { {  INT64_C( 2379025802943267314) },
      { -INT64_C(                   3) },
      {  INT64_C(  297378225367908414) } },
    { { -INT64_C( 7880843795017100943) },
      {  INT64_C(                  13) },
      {                      INT64_MIN } },
    { {  INT64_C( 5060086036473607234) },
      { -INT64_C(                  12) },
      {  INT64_C(    1235372567498439) } },
    { { -INT64_C( 1251288881944570923) },
      {  INT64_C(                  59) },
      {                      INT64_MIN } },
    { { -INT64_C(  818197169458855311) },
      { -INT64_C(                  46) },
      { -INT64_C(               11627) } },
    { { -INT64_C( 3470264563169298453) },
      { -INT64_C(                  56) },
      { -INT64_C(                  48) } },
    { { -INT64_C( 6566310065777805324) },
      { -INT64_C(                  54) },
      { -INT64_C(                 365) } },
    { {  INT64_C( 8846445702649439056) },
      { -INT64_C(                   5) },
      {  INT64_C(  276451428207794971) } },
    { {                      INT64_MAX },
      { -INT64_C(                   1) },
      {  INT64_C( 4611686018427387904) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int64x1_t a = simde_vld1_s64(test_vec[i].a);
    simde_int64x1_t b = simde_vld1_s64(test_vec[i].b);
    simde_int64x1_t r = simde_vqrshl_s64(a, b);

    simde_test_arm_neon_assert_equal_i64x1(r, simde_vld1_s64(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int64x1_t a = simde_test_arm_neon_random_i64x1();
    simde_int64x1_t b = simde_test_arm_neon_random_i64x1();
    simde_int64x1_t r = simde_vqrshl_s64(a, b);

    simde_test_arm_neon_write_i64x1(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i64x1(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i64x1(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshl_u8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint8_t a[8];
    int8_t b[8];
    uint8_t r[8];
  } test_vec[] = {
    { {  UINT8_C(    17),  UINT8_C(   254),  UINT8_C(   166),  UINT8_C(   242),
         UINT8_C(   224),  UINT8_C(    96),  UINT8_C(   134),  UINT8_C(    31) },
      { -INT8_C(     2),  INT8_C(     6), -INT8_C(     5), -INT8_C(     2),
        -INT8_C(     1),  INT8_C(     0), -INT8_C(     4),  INT8_C(     4) },
      {  UINT8_C(     4),  UINT8_C(   255),  UINT8_C(     5),  UINT8_C(    61),
         UINT8_C(   112),  UINT8_C(    96),  UINT8_C(     8),  UINT8_C(   255) } },
    { {  UINT8_C(   236),  UINT8_C(    17),  UINT8_C(   172),  UINT8_C(    17),
         UINT8_C(    72),  UINT8_C(    97),  UINT8_C(   108),  UINT8_C(   128) },
      {  INT8_C(     6),  INT8_C(     0),  INT8_C(     6),  INT8_C(     7),
        -INT8_C(     4), -INT8_C(     5),  INT8_C(     1), -INT8_C(     7) },
      {  UINT8_C(   255),  UINT8_C(    17),  UINT8_C(   255),  UINT8_C(   255),
         UINT8_C(     5),  UINT8_C(     3),  UINT8_C(   216),  UINT8_C(     1) } },
    { {  UINT8_C(   101),  UINT8_C(   167),  UINT8_C(    60),  UINT8_C(   143),
         UINT8_C(   234),  UINT8_C(   177),  UINT8_C(   216),  UINT8_C(   221) },
      { -INT8_C(     3), -INT8_C(     8),  INT8_C(     3), -INT8_C(     6),
         INT8_C(     3),  INT8_C(     5), -INT8_C(     8),  INT8_C(     5) },
      {  UINT8_C(    13),  UINT8_C(     1),  UINT8_C(   255),  UINT8_C(     2),
         UINT8_C(   255),  UINT8_C(   255),  UINT8_C(     1),  UINT8_C(   255) } },
    { {  UINT8_C(     9),  UINT8_C(   223),  UINT8_C(    27),  UINT8_C(   204),
         UINT8_C(   106),  UINT8_C(   241),  UINT8_C(    32),  UINT8_C(   220) },
      { -INT8_C(     5),  INT8_C(     8),  INT8_C(     3),  INT8_C(     4),
        -INT8_C(     4),  INT8_C(     6),  INT8_C(     6), -INT8_C(     8) },
      {  UINT8_C(     0),  UINT8_C(   255),  UINT8_C(   216),  UINT8_C(   255),
         UINT8_C(     7),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(     1) } },
    { {  UINT8_C(   175),  UINT8_C(   114),  UINT8_C(    53),  UINT8_C(    46),
         UINT8_C(    26),  UINT8_C(   170),  UINT8_C(   171),  UINT8_C(   213) },
      { -INT8_C(     6), -INT8_C(     6), -INT8_C(     4),  INT8_C(     3),
         INT8_C(     1), -INT8_C(     2), -INT8_C(     3), -INT8_C(     8) },
      {  UINT8_C(     3),  UINT8_C(     2),  UINT8_C(     3),  UINT8_C(   255),
         UINT8_C(    52),  UINT8_C(    43),  UINT8_C(    21),  UINT8_C(     1) } },
    { {  UINT8_C(    37),  UINT8_C(   113),  UINT8_C(   207),  UINT8_C(   209),
         UINT8_C(   209),  UINT8_C(     8),  UINT8_C(   239),  UINT8_C(    42) },
      { -INT8_C(     8), -INT8_C(     6), -INT8_C(     7), -INT8_C(     5),
         INT8_C(     2),  INT8_C(     7),  INT8_C(     1), -INT8_C(     5) },
      {  UINT8_C(     0),  UINT8_C(     2),  UINT8_C(     2),  UINT8_C(     7),
         UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(     1) } },
    { {  UINT8_C(   249),  UINT8_C(    98),  UINT8_C(    62),  UINT8_C(   112),
         UINT8_C(    21),  UINT8_C(    89),  UINT8_C(   159),  UINT8_C(   208) },
      {  INT8_C(     4), -INT8_C(     3),  INT8_C(     8),  INT8_C(     0),
        -INT8_C(     4), -INT8_C(     6), -INT8_C(     3),  INT8_C(     2) },
      {  UINT8_C(   255),  UINT8_C(    12),  UINT8_C(   255),  UINT8_C(   112),
         UINT8_C(     1),  UINT8_C(     1),  UINT8_C(    20),  UINT8_C(   255) } },
    { {  UINT8_C(    38),  UINT8_C(   128),  UINT8_C(   239),  UINT8_C(   139),
         UINT8_C(   165),  UINT8_C(   233),  UINT8_C(   134),  UINT8_C(   247) },
      { -INT8_C(     3),  INT8_C(     5),  INT8_C(     6),  INT8_C(     3),
        -INT8_C(     5),  INT8_C(     4),  INT8_C(     1),  INT8_C(     7) },
      {  UINT8_C(     5),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   255),
         UINT8_C(     5),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   255) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint8x8_t a = simde_vld1_u8(test_vec[i].a);
    simde_int8x8_t b = simde_vld1_s8(test_vec[i].b);
    simde_uint8x8_t r = simde_vqrshl_u8(a, b);

    simde_test_arm_neon_assert_equal_u8x8(r, simde_vld1_u8(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint8x8_t a = simde_test_arm_neon_random_u8x8();
    simde_int8x8_t b = simde_test_arm_neon_random_i8x8();
    simde_uint8x8_t r = simde_vqrshl_u8(a, b);

    simde_test_arm_neon_write_u8x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i8x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u8x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshl_u16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint16_t a[4];
    int16_t b[4];
    uint16_t r[4];
  } test_vec[] = {
    { {  UINT16_C(   36529),  UINT16_C(   29829),  UINT16_C(    2096),  UINT16_C(    2729) },
      {  INT16_C(       0),  INT16_C(       4),  INT16_C(       9), -INT16_C(       4) },
      {  UINT16_C(   36529),  UINT16_C(   65535),  UINT16_C(   65535),  UINT16_C(     171) } },
    { {  UINT16_C(   39946),  UINT16_C(   27950),  UINT16_C(   23605),  UINT16_C(    4911) },
      {  INT16_C(      11), -INT16_C(      14), -INT16_C(       7),  INT16_C(       9) },
      {  UINT16_C(   65535),  UINT16_C(       2),  UINT16_C(     184),  UINT16_C(   65535) } },
    { {  UINT16_C(   42859),  UINT16_C(    2240),  UINT16_C(   20043),  UINT16_C(   10036) },
      {  INT16_C(       8),  INT16_C(      14), -INT16_C(      15), -INT16_C(      12) },
      {  UINT16_C(   65535),  UINT16_C(   65535),  UINT16_C(       1),  UINT16_C(       2) } },
    { {  UINT16_C(   50980),  UINT16_C(     165),  UINT16_C(   10039),  UINT16_C(   30538) },
      {  INT16_C(       8), -INT16_C(       8),  INT16_C(      11), -INT16_C(       9) },
      {  UINT16_C(   65535),  UINT16_C(       1),  UINT16_C(   65535),  UINT16_C(      60) } },
    { {  UINT16_C(   61605),  UINT16_C(   27801),  UINT16_C(   14514),  UINT16_C(   31459) },
      { -INT16_C(       2),  INT16_C(       9),  INT16_C(       7),  INT16_C(      13) },
      {  UINT16_C(   15401),  UINT16_C(   65535),  UINT16_C(   65535),  UINT16_C(   65535) } },
    { {  UINT16_C(   27895),  UINT16_C(   11683),  UINT16_C(   16781),  UINT16_C(   38533) },
      { -INT16_C(      10), -INT16_C(       4),  INT16_C(      16), -INT16_C(      10) },
      {  UINT16_C(      27),  UINT16_C(     730),  UINT16_C(   65535),  UINT16_C(      38) } },
    { {  UINT16_C(   53996),  UINT16_C(   31874),  UINT16_C(   57883),  UINT16_C(    4747) },
      { -INT16_C(      16),  INT16_C(       8), -INT16_C(       9),  INT16_C(       9) },
      {  UINT16_C(       1),  UINT16_C(   65535),  UINT16_C(     113),  UINT16_C(   65535) } },
    { {  UINT16_C(   37059),  UINT16_C(   57524),  UINT16_C(   51263),  UINT16_C(   61052) },
      {  INT16_C(       6), -INT16_C(      11), -INT16_C(       1), -INT16_C(       6) },
      {  UINT16_C(   65535),  UINT16_C(      28),  UINT16_C(   25632),  UINT16_C(     954) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint16x4_t a = simde_vld1_u16(test_vec[i].a);
    simde_int16x4_t b = simde_vld1_s16(test_vec[i].b);
    simde_uint16x4_t r = simde_vqrshl_u16(a, b);
    simde_test_arm_neon_assert_equal_u16x4(r, simde_vld1_u16(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint16x4_t a = simde_test_arm_neon_random_u16x4();
    simde_int16x4_t b = simde_test_arm_neon_random_i16x4();
    simde_uint16x4_t r = simde_vqrshl_u16(a, b);

    simde_test_arm_neon_write_u16x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i16x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u16x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshl_u32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint32_t a[2];
    int32_t b[2];
    uint32_t r[2];
  } test_vec[] = {
    { {  UINT32_C(  4032142366),  UINT32_C(  2503991721) },
      {  INT32_C(          23), -INT32_C(          11) },
      {  UINT32_C(  4294967295),  UINT32_C(     1222652) } },
    { {  UINT32_C(  1849889977),  UINT32_C(  1539239213) },
      {  INT32_C(           3), -INT32_C(          31) },
      {  UINT32_C(  4294967295),  UINT32_C(           1) } },
    { {  UINT32_C(  3780982194),  UINT32_C(  1551554012) },
      {  INT32_C(           8),  INT32_C(          30) },
      {  UINT32_C(  4294967295),  UINT32_C(  4294967295) } },
    { {  UINT32_C(   457931350),  UINT32_C(  2322708363) },
      { -INT32_C(          24), -INT32_C(          15) },
      {  UINT32_C(          27),  UINT32_C(       70883) } },
    { {  UINT32_C(  1302772324),  UINT32_C(  3149457440) },
      { -INT32_C(           8),  INT32_C(          31) },
      {  UINT32_C(     5088954),  UINT32_C(  4294967295) } },
    { {  UINT32_C(  2656566271),  UINT32_C(    10727285) },
      { -INT32_C(          11),  INT32_C(          23) },
      {  UINT32_C(     1297151),  UINT32_C(  4294967295) } },
    { {  UINT32_C(   580186040),  UINT32_C(   879645142) },
      {  INT32_C(          26),  INT32_C(           8) },
      {  UINT32_C(  4294967295),  UINT32_C(  4294967295) } },
    { {  UINT32_C(  2842364758),  UINT32_C(   627566619) },
      { -INT32_C(          19), -INT32_C(          17) },
      {  UINT32_C(        5421),  UINT32_C(        4788) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint32x2_t a = simde_vld1_u32(test_vec[i].a);
    simde_int32x2_t b = simde_vld1_s32(test_vec[i].b);
    simde_uint32x2_t r = simde_vqrshl_u32(a, b);
    simde_test_arm_neon_assert_equal_u32x2(r, simde_vld1_u32(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint32x2_t a = simde_test_arm_neon_random_u32x2();
    simde_int32x2_t b = simde_test_arm_neon_random_i32x2();
    simde_uint32x2_t r = simde_vqrshl_u32(a, b);

    simde_test_arm_neon_write_u32x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i32x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u32x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshl_u64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint64_t a[1];
    int64_t b[1];
    uint64_t r[1];
  } test_vec[] = {
    { {  UINT64_C(10606652271688138073) },
      { -INT64_C(                  24) },
      {  UINT64_C(        632205740910) } },
    { {  UINT64_C( 6951000685328207734) },
      { -INT64_C(                  41) },
      {  UINT64_C(             3160949) } },
    { {  UINT64_C( 2652753700446653166) },
      {  INT64_C(                  16) },
      {  UINT64_C(18446744073709551615) } },
    { {  UINT64_C( 2887850072060195408) },
      {  INT64_C(                   8) },
      {  UINT64_C(18446744073709551615) } },
    { {  UINT64_C( 3003532981166020057) },
      { -INT64_C(                  18) },
      {  UINT64_C(      11457569050469) } },
    { {  UINT64_C( 2467308484911859140) },
      { -INT64_C(                   4) },
      {  UINT64_C(  154206780306991196) } },
    { {  UINT64_C(10329885492628171968) },
      {  INT64_C(                   8) },
      {  UINT64_C(18446744073709551615) } },
    { {  UINT64_C( 2016469675685954892) },
      { -INT64_C(                  62) },
      {  UINT64_C(                   0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint64x1_t a = simde_vld1_u64(test_vec[i].a);
    simde_int64x1_t b = simde_vld1_s64(test_vec[i].b);
    simde_uint64x1_t r = simde_vqrshl_u64(a, b);
    simde_test_arm_neon_assert_equal_u64x1(r, simde_vld1_u64(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint64x1_t a = simde_test_arm_neon_random_u64x1();
    simde_int64x1_t b = simde_test_arm_neon_random_i64x1();
    simde_uint64x1_t r = simde_vqrshl_u64(a, b);

    simde_test_arm_neon_write_u64x1(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i64x1(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u64x1(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_s8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int8_t a[16];
    int8_t b[16];
    int8_t r[16];
  } test_vec[] = {
    { {  INT8_C(   40), -INT8_C(   12), -INT8_C(  107),  INT8_C(   91),
         INT8_C(   39),  INT8_C(   70), -INT8_C(   66), -INT8_C(   32),
         INT8_C(   44),  INT8_C(  106),  INT8_C(   13), -INT8_C(   17),
         INT8_C(   42), -INT8_C(   33),  INT8_C(    0),  INT8_C(   99) },
      { -INT8_C(    1),  INT8_C(    5),  INT8_C(    7), -INT8_C(    6),
         INT8_C(    5),  INT8_C(    3), -INT8_C(    4), -INT8_C(    2),
         INT8_C(    8),  INT8_C(    7), -INT8_C(    8), -INT8_C(    8),
         INT8_C(    8),  INT8_C(    4),  INT8_C(    4), -INT8_C(    7) },
      {  INT8_C(   20),       INT8_MIN,       INT8_MIN,  INT8_C(    1),
              INT8_MAX,       INT8_MAX, -INT8_C(    4), -INT8_C(    8),
              INT8_MAX,       INT8_MAX,  INT8_C(    0),  INT8_C(    0),
              INT8_MAX,       INT8_MIN,  INT8_C(    0),  INT8_C(    1) } },
    { {  INT8_C(   13), -INT8_C(   18), -INT8_C(  109),  INT8_C(  117),
        -INT8_C(  115),  INT8_C(   11),  INT8_C(   28),  INT8_C(   34),
         INT8_C(   84),  INT8_C(   24), -INT8_C(   56),  INT8_C(   69),
         INT8_C(  112), -INT8_C(   93), -INT8_C(   57),  INT8_C(   46) },
      {  INT8_C(    3),  INT8_C(    1), -INT8_C(    6),  INT8_C(    4),
        -INT8_C(    7),  INT8_C(    1),  INT8_C(    3), -INT8_C(    4),
         INT8_C(    6),  INT8_C(    1), -INT8_C(    2), -INT8_C(    3),
         INT8_C(    0),  INT8_C(    6),  INT8_C(    3),  INT8_C(    8) },
      {  INT8_C(  104), -INT8_C(   36), -INT8_C(    2),       INT8_MAX,
        -INT8_C(    1),  INT8_C(   22),       INT8_MAX,  INT8_C(    2),
              INT8_MAX,  INT8_C(   48), -INT8_C(   14),  INT8_C(    9),
         INT8_C(  112),       INT8_MIN,       INT8_MIN,       INT8_MAX } },
    { { -INT8_C(   20),  INT8_C(   92), -INT8_C(  120),  INT8_C(   62),
        -INT8_C(   85),  INT8_C(   61), -INT8_C(    9), -INT8_C(  121),
         INT8_C(   18), -INT8_C(   16), -INT8_C(  124), -INT8_C(  109),
         INT8_C(  101),  INT8_C(   14), -INT8_C(   15), -INT8_C(   69) },
      {  INT8_C(    2), -INT8_C(    3),  INT8_C(    3),  INT8_C(    2),
        -INT8_C(    2), -INT8_C(    2), -INT8_C(    2),  INT8_C(    6),
         INT8_C(    0),  INT8_C(    2),  INT8_C(    4),  INT8_C(    5),
        -INT8_C(    8), -INT8_C(    3), -INT8_C(    1),  INT8_C(    8) },
      { -INT8_C(   80),  INT8_C(   12),       INT8_MIN,       INT8_MAX,
        -INT8_C(   21),  INT8_C(   15), -INT8_C(    2),       INT8_MIN,
         INT8_C(   18), -INT8_C(   64),       INT8_MIN,       INT8_MIN,
         INT8_C(    0),  INT8_C(    2), -INT8_C(    7),       INT8_MIN } },
    { {  INT8_C(   38),  INT8_C(   60), -INT8_C(   70),  INT8_C(   91),
         INT8_C(  120), -INT8_C(  121),  INT8_C(   71), -INT8_C(   22),
         INT8_C(   92),  INT8_C(    8),  INT8_C(  124), -INT8_C(  117),
         INT8_C(  115),  INT8_C(   44), -INT8_C(   93),  INT8_C(   66) },
      {  INT8_C(    1),  INT8_C(    5),  INT8_C(    2),  INT8_C(    8),
        -INT8_C(    6), -INT8_C(    4),  INT8_C(    8), -INT8_C(    2),
         INT8_C(    5),  INT8_C(    4), -INT8_C(    2),  INT8_C(    2),
         INT8_C(    8),  INT8_C(    0), -INT8_C(    5),  INT8_C(    0) },
      {  INT8_C(   76),       INT8_MAX,       INT8_MIN,       INT8_MAX,
         INT8_C(    2), -INT8_C(    8),       INT8_MAX, -INT8_C(    5),
              INT8_MAX,       INT8_MAX,  INT8_C(   31),       INT8_MIN,
              INT8_MAX,  INT8_C(   44), -INT8_C(    3),  INT8_C(   66) } },
    { {  INT8_C(  122), -INT8_C(  112), -INT8_C(  124),  INT8_C(   42),
         INT8_C(   87), -INT8_C(  117),  INT8_C(   56), -INT8_C(   93),
         INT8_C(   78),  INT8_C(   31), -INT8_C(    3), -INT8_C(   50),
         INT8_C(  116),  INT8_C(   68),  INT8_C(   64),  INT8_C(   69) },
      {  INT8_C(    2),  INT8_C(    4), -INT8_C(    7),  INT8_C(    4),
        -INT8_C(    3), -INT8_C(    8),  INT8_C(    8),  INT8_C(    1),
        -INT8_C(    5), -INT8_C(    1),  INT8_C(    6), -INT8_C(    3),
        -INT8_C(    3),  INT8_C(    5),  INT8_C(    2),  INT8_C(    1) },
      {       INT8_MAX,       INT8_MIN, -INT8_C(    1),       INT8_MAX,
         INT8_C(   11),  INT8_C(    0),       INT8_MAX,       INT8_MIN,
         INT8_C(    2),  INT8_C(   16),       INT8_MIN, -INT8_C(    6),
         INT8_C(   15),       INT8_MAX,       INT8_MAX,       INT8_MAX } },
    { { -INT8_C(   34), -INT8_C(    5), -INT8_C(   85), -INT8_C(   77),
         INT8_C(   98),  INT8_C(  107),  INT8_C(   84), -INT8_C(  115),
        -INT8_C(  127), -INT8_C(    1), -INT8_C(   33),  INT8_C(  111),
         INT8_C(   33),  INT8_C(   97),  INT8_C(   46),  INT8_C(   76) },
      {  INT8_C(    8),  INT8_C(    4),  INT8_C(    0), -INT8_C(    7),
         INT8_C(    2), -INT8_C(    6),  INT8_C(    4), -INT8_C(    8),
         INT8_C(    4),  INT8_C(    6),  INT8_C(    4), -INT8_C(    8),
         INT8_C(    1), -INT8_C(    5), -INT8_C(    6), -INT8_C(    7) },
      {       INT8_MIN, -INT8_C(   80), -INT8_C(   85), -INT8_C(    1),
              INT8_MAX,  INT8_C(    2),       INT8_MAX,  INT8_C(    0),
              INT8_MIN, -INT8_C(   64),       INT8_MIN,  INT8_C(    0),
         INT8_C(   66),  INT8_C(    3),  INT8_C(    1),  INT8_C(    1) } },
    { { -INT8_C(   69), -INT8_C(   87),  INT8_C(   96),  INT8_C(   21),
         INT8_C(   66), -INT8_C(   29), -INT8_C(   72), -INT8_C(  116),
        -INT8_C(  125),  INT8_C(   72),  INT8_C(   98),  INT8_C(   60),
         INT8_C(   99),  INT8_C(  102), -INT8_C(   72), -INT8_C(   39) },
      {  INT8_C(    2), -INT8_C(    3), -INT8_C(    6), -INT8_C(    3),
        -INT8_C(    3),  INT8_C(    4),  INT8_C(    4), -INT8_C(    2),
         INT8_C(    0), -INT8_C(    5),  INT8_C(    1),  INT8_C(    5),
        -INT8_C(    3), -INT8_C(    3), -INT8_C(    7), -INT8_C(    7) },
      {       INT8_MIN, -INT8_C(   11),  INT8_C(    2),  INT8_C(    3),
         INT8_C(    8),       INT8_MIN,       INT8_MIN, -INT8_C(   29),
        -INT8_C(  125),  INT8_C(    2),       INT8_MAX,       INT8_MAX,
         INT8_C(   12),  INT8_C(   13), -INT8_C(    1),  INT8_C(    0) } },
    { { -INT8_C(  121),  INT8_C(   38),  INT8_C(  117),  INT8_C(   37),
        -INT8_C(   14), -INT8_C(   96),  INT8_C(   74),  INT8_C(  115),
        -INT8_C(   24),  INT8_C(   64),  INT8_C(  105), -INT8_C(  122),
        -INT8_C(   82),  INT8_C(    2),  INT8_C(  119),  INT8_C(   84) },
      {  INT8_C(    5),  INT8_C(    1),  INT8_C(    1),  INT8_C(    3),
        -INT8_C(    1),  INT8_C(    3),  INT8_C(    8),  INT8_C(    2),
         INT8_C(    0), -INT8_C(    4),  INT8_C(    8),  INT8_C(    3),
         INT8_C(    0),  INT8_C(    0), -INT8_C(    6), -INT8_C(    1) },
      {       INT8_MIN,  INT8_C(   76),       INT8_MAX,       INT8_MAX,
        -INT8_C(    7),       INT8_MIN,       INT8_MAX,       INT8_MAX,
        -INT8_C(   24),  INT8_C(    4),       INT8_MAX,       INT8_MIN,
        -INT8_C(   82),  INT8_C(    2),  INT8_C(    2),  INT8_C(   42) } },
    { {       INT8_MAX,       INT8_MAX,       INT8_MAX,       INT8_MAX,
              INT8_MAX,       INT8_MAX,       INT8_MAX,       INT8_MAX,
              INT8_MAX,       INT8_MAX,       INT8_MAX,       INT8_MAX,
              INT8_MAX,       INT8_MAX,       INT8_MAX,       INT8_MAX },
      { -INT8_C(    1), -INT8_C(    1), -INT8_C(    1), -INT8_C(    1),
        -INT8_C(    1), -INT8_C(    1), -INT8_C(    1), -INT8_C(    1),
        -INT8_C(    1), -INT8_C(    1), -INT8_C(    1), -INT8_C(    1),
        -INT8_C(    1), -INT8_C(    1), -INT8_C(    1), -INT8_C(    1) },
      {  INT8_C(   64),  INT8_C(   64),  INT8_C(   64),  INT8_C(   64),
         INT8_C(   64),  INT8_C(   64),  INT8_C(   64),  INT8_C(   64),
         INT8_C(   64),  INT8_C(   64),  INT8_C(   64),  INT8_C(   64),
         INT8_C(   64),  INT8_C(   64),  INT8_C(   64),  INT8_C(   64) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int8x16_t a = simde_vld1q_s8(test_vec[i].a);
    simde_int8x16_t b = simde_vld1q_s8(test_vec[i].b);
    simde_int8x16_t r = simde_vqrshlq_s8(a, b);

    simde_test_arm_neon_assert_equal_i8x16(r, simde_vld1q_s8(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int8x16_t a = simde_test_arm_neon_random_i8x16();
    simde_int8x16_t b = simde_test_arm_neon_random_i8x16();
    simde_int8x16_t r = simde_vqrshlq_s8(a, b);

    simde_test_arm_neon_write_i8x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i8x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i8x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_s16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int16_t a[8];
    int16_t b[8];
    int16_t r[8];
  } test_vec[] = {
    { {  INT16_C(      17), -INT16_C(   18060),  INT16_C(    2082),  INT16_C(    3026),
        -INT16_C(   23678),  INT16_C(   13144), -INT16_C(   19898),  INT16_C(   26465) },
      { -INT16_C(       9),  INT16_C(       4), -INT16_C(       9), -INT16_C(       7),
         INT16_C(       6), -INT16_C(       8), -INT16_C(       5), -INT16_C(       7) },
      {  INT16_C(       0),          INT16_MIN,  INT16_C(       4),  INT16_C(      24),
                 INT16_MIN,  INT16_C(      51), -INT16_C(     622),  INT16_C(     207) } },
    { { -INT16_C(   31358),  INT16_C(   28497), -INT16_C(   24101), -INT16_C(   29556),
         INT16_C(   23640),  INT16_C(   26523),  INT16_C(   30555),  INT16_C(   32733) },
      {  INT16_C(       8),  INT16_C(      12),  INT16_C(      11), -INT16_C(       9),
        -INT16_C(       8), -INT16_C(      11),  INT16_C(       4), -INT16_C(       3) },
      {          INT16_MIN,          INT16_MAX,          INT16_MIN, -INT16_C(      58),
         INT16_C(      92),  INT16_C(      13),          INT16_MAX,  INT16_C(    4092) } },
    { { -INT16_C(   25371), -INT16_C(   28626), -INT16_C(   10661), -INT16_C(   29662),
         INT16_C(    8039),  INT16_C(   27599),  INT16_C(   27158),  INT16_C(   14531) },
      {  INT16_C(       0), -INT16_C(      10),  INT16_C(       2),  INT16_C(      11),
        -INT16_C(       2), -INT16_C(       9),  INT16_C(      15),  INT16_C(       6) },
      { -INT16_C(   25371), -INT16_C(      28),          INT16_MIN,          INT16_MIN,
         INT16_C(    2010),  INT16_C(      54),          INT16_MAX,          INT16_MAX } },
    { {  INT16_C(   29383),  INT16_C(   13775),  INT16_C(    1669),  INT16_C(   16370),
        -INT16_C(   31950), -INT16_C(   10488), -INT16_C(    4281), -INT16_C(   15443) },
      {  INT16_C(       6), -INT16_C(      14),  INT16_C(       6), -INT16_C(      13),
        -INT16_C(       7),  INT16_C(       5),  INT16_C(      12), -INT16_C(       2) },
      {          INT16_MAX,  INT16_C(       1),          INT16_MAX,  INT16_C(       2),
        -INT16_C(     250),          INT16_MIN,          INT16_MIN, -INT16_C(    3861) } },
    { { -INT16_C(   16008),  INT16_C(    6576),  INT16_C(   25375), -INT16_C(    2472),
        -INT16_C(   22062), -INT16_C(    1594), -INT16_C(      33), -INT16_C(   29140) },
      { -INT16_C(      10), -INT16_C(      16), -INT16_C(      14),  INT16_C(      15),
        -INT16_C(      11), -INT16_C(      13),  INT16_C(       3), -INT16_C(      13) },
      { -INT16_C(      16),  INT16_C(       0),  INT16_C(       2),          INT16_MIN,
        -INT16_C(      11),  INT16_C(       0), -INT16_C(     264), -INT16_C(       4) } },
    { { -INT16_C(   17438),  INT16_C(   21035), -INT16_C(    4482), -INT16_C(    5337),
        -INT16_C(   25235), -INT16_C(    6754), -INT16_C(    9668), -INT16_C(   27862) },
      { -INT16_C(       1), -INT16_C(      15),  INT16_C(       5), -INT16_C(       4),
        -INT16_C(       5),  INT16_C(      14), -INT16_C(       8), -INT16_C(       9) },
      { -INT16_C(    8719),  INT16_C(       1),          INT16_MIN, -INT16_C(     334),
        -INT16_C(     789),          INT16_MIN, -INT16_C(      38), -INT16_C(      54) } },
    { {  INT16_C(   10172), -INT16_C(    6237), -INT16_C(   25431), -INT16_C(   30731),
        -INT16_C(    2806), -INT16_C(   24307), -INT16_C(    1303), -INT16_C(   17051) },
      { -INT16_C(       4),  INT16_C(      13), -INT16_C(      11), -INT16_C(       1),
         INT16_C(      10),  INT16_C(       7), -INT16_C(      16),  INT16_C(       1) },
      {  INT16_C(     636),          INT16_MIN, -INT16_C(      12), -INT16_C(   15365),
                 INT16_MIN,          INT16_MIN,  INT16_C(       0),          INT16_MIN } },
    { { -INT16_C(    1297),  INT16_C(    6102),  INT16_C(   12306),  INT16_C(   23191),
        -INT16_C(    2630), -INT16_C(   28109), -INT16_C(   20844),  INT16_C(     271) },
      {  INT16_C(       3),  INT16_C(      15),  INT16_C(       8),  INT16_C(       6),
         INT16_C(       0),  INT16_C(      15), -INT16_C(       5), -INT16_C(      13) },
      { -INT16_C(   10376),          INT16_MAX,          INT16_MAX,          INT16_MAX,
        -INT16_C(    2630),          INT16_MIN, -INT16_C(     651),  INT16_C(       0) } },
    { {          INT16_MAX,          INT16_MAX,          INT16_MAX,          INT16_MAX,
                 INT16_MAX,          INT16_MAX,          INT16_MAX,          INT16_MAX },
      { -INT16_C(       1), -INT16_C(       1), -INT16_C(       1), -INT16_C(       1),
        -INT16_C(       1), -INT16_C(       1), -INT16_C(       1), -INT16_C(       1) },
      {  INT16_C(   16384),  INT16_C(   16384),  INT16_C(   16384),  INT16_C(   16384),
         INT16_C(   16384),  INT16_C(   16384),  INT16_C(   16384),  INT16_C(   16384) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int16x8_t a = simde_vld1q_s16(test_vec[i].a);
    simde_int16x8_t b = simde_vld1q_s16(test_vec[i].b);
    simde_int16x8_t r = simde_vqrshlq_s16(a, b);

    simde_test_arm_neon_assert_equal_i16x8(r, simde_vld1q_s16(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int16x8_t a = simde_test_arm_neon_random_i16x8();
    simde_int16x8_t b = simde_test_arm_neon_random_i16x8();
    simde_int16x8_t r = simde_vqrshlq_s16(a, b);

    simde_test_arm_neon_write_i16x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i16x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_s32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int32_t a[4];
    int32_t b[4];
    int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(  1814886595), -INT32_C(  1646956830),  INT32_C(  1237502269),  INT32_C(   534733024) },
      { -INT32_C(          21), -INT32_C(          25),  INT32_C(          31), -INT32_C(          11) },
      {  INT32_C(         865), -INT32_C(          49),              INT32_MAX,  INT32_C(      261100) } },
    { {  INT32_C(  1736113552),  INT32_C(  1258318563),  INT32_C(  1069774949),  INT32_C(   755812542) },
      { -INT32_C(          32),  INT32_C(           8),  INT32_C(          32),  INT32_C(          19) },
      {  INT32_C(           0),              INT32_MAX,              INT32_MAX,              INT32_MAX } },
    { { -INT32_C(  1923447339), -INT32_C(  1220148547), -INT32_C(  1485011002), -INT32_C(  1143271085) },
      { -INT32_C(          30), -INT32_C(          24), -INT32_C(          25),  INT32_C(          30) },
      { -INT32_C(           2), -INT32_C(          73), -INT32_C(          44),              INT32_MIN } },
    { {  INT32_C(  1865665830),  INT32_C(  2143927441), -INT32_C(  1135678127),  INT32_C(   123259338) },
      {  INT32_C(          20), -INT32_C(           9), -INT32_C(          31), -INT32_C(          14) },
      {              INT32_MAX,  INT32_C(     4187358), -INT32_C(           1),  INT32_C(        7523) } },
    { {  INT32_C(     3352422), -INT32_C(   698249524),  INT32_C(  1806447909),  INT32_C(   800823842) },
      { -INT32_C(           6),  INT32_C(           4), -INT32_C(           6),  INT32_C(          32) },
      {  INT32_C(       52382),              INT32_MIN,  INT32_C(    28225749),              INT32_MAX } },
    { { -INT32_C(   337342704),  INT32_C(  1303521110),  INT32_C(   425408117),  INT32_C(   231996299) },
      { -INT32_C(          32), -INT32_C(          16), -INT32_C(          31),  INT32_C(           2) },
      {  INT32_C(           0),  INT32_C(       19890),  INT32_C(           0),  INT32_C(   927985196) } },
    { { -INT32_C(   529877133), -INT32_C(  1960225014), -INT32_C(  1757285913),  INT32_C(  1764816475) },
      {  INT32_C(          20), -INT32_C(           6),  INT32_C(          32),  INT32_C(          24) },
      {              INT32_MIN, -INT32_C(    30628516),              INT32_MIN,              INT32_MAX } },
    { {  INT32_C(   219035158),  INT32_C(  2118934887),  INT32_C(   641686675),  INT32_C(   264846024) },
      {  INT32_C(           6),  INT32_C(          28), -INT32_C(           1),  INT32_C(          12) },
      {              INT32_MAX,              INT32_MAX,  INT32_C(   320843338),              INT32_MAX } },
    { {              INT32_MAX,              INT32_MAX,              INT32_MAX,              INT32_MAX },
      { -INT32_C(           1), -INT32_C(           1), -INT32_C(           1), -INT32_C(           1) },
      {  INT32_C(  1073741824),  INT32_C(  1073741824),  INT32_C(  1073741824),  INT32_C(  1073741824) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int32x4_t a = simde_vld1q_s32(test_vec[i].a);
    simde_int32x4_t b = simde_vld1q_s32(test_vec[i].b);
    simde_int32x4_t r = simde_vqrshlq_s32(a, b);
    simde_test_arm_neon_assert_equal_i32x4(r, simde_vld1q_s32(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int32x4_t a = simde_test_arm_neon_random_i32x4();
    simde_int32x4_t b = simde_test_arm_neon_random_i32x4();
    simde_int32x4_t r = simde_vqrshlq_s32(a, b);

    simde_test_arm_neon_write_i32x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_s64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    int64_t a[2];
    int64_t b[2];
    int64_t r[2];
  } test_vec[] = {
    { { -INT64_C( 5414884328532004708), -INT64_C( 7732031084687741673) },
      { -INT64_C(                  63),  INT64_C(                  61) },
      { -INT64_C(                   1),                      INT64_MIN } },
    { {  INT64_C( 3932969372243887233), -INT64_C( 7443478377457562434) },
      { -INT64_C(                  45), -INT64_C(                   1) },
      {  INT64_C(              111782), -INT64_C( 3721739188728781217) } },
    { { -INT64_C( 4624332781887269652),  INT64_C( 9194359439567769551) },
      {  INT64_C(                  18),  INT64_C(                  12) },
      {                      INT64_MIN,                      INT64_MAX } },
    { {  INT64_C( 1303116637892605202),  INT64_C( 2411425157371539502) },
      { -INT64_C(                  35), -INT64_C(                  12) },
      {  INT64_C(            37925686),  INT64_C(     588726845061411) } },
    { { -INT64_C( 5953546370211056151),  INT64_C( 4807810721811645268) },
      { -INT64_C(                  15),  INT64_C(                  44) },
      { -INT64_C(     181687816473726),                      INT64_MAX } },
    { { -INT64_C(  671209687377174666), -INT64_C( 2105758642682840471) },
      {  INT64_C(                  35),  INT64_C(                   6) },
      {                      INT64_MIN,                      INT64_MIN } },
    { { -INT64_C( 1318546396640194493),  INT64_C( 4726180957159977633) },
      {  INT64_C(                  33),  INT64_C(                   3) },
      {                      INT64_MIN,                      INT64_MAX } },
    { {  INT64_C(  570160631599597498),  INT64_C( 7291438976119207855) },
      { -INT64_C(                  54),  INT64_C(                  30) },
      {  INT64_C(                  32),                      INT64_MAX } },
    { {                      INT64_MAX,                      INT64_MAX },
      { -INT64_C(                   1), -INT64_C(                   1) },
      {  INT64_C( 4611686018427387904),  INT64_C( 4611686018427387904) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int64x2_t a = simde_vld1q_s64(test_vec[i].a);
    simde_int64x2_t b = simde_vld1q_s64(test_vec[i].b);
    simde_int64x2_t r = simde_vqrshlq_s64(a, b);
    simde_test_arm_neon_assert_equal_i64x2(r, simde_vld1q_s64(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_int64x2_t a = simde_test_arm_neon_random_i64x2();
    simde_int64x2_t b = simde_test_arm_neon_random_i64x2();
    simde_int64x2_t r = simde_vqrshlq_s64(a, b);

    simde_test_arm_neon_write_i64x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_i64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_u8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint8_t a[16];
    int8_t b[16];
    uint8_t r[16];
  } test_vec[] = {
    { {  UINT8_C(    48),  UINT8_C(   110),  UINT8_C(   160),  UINT8_C(   230),
         UINT8_C(   244),  UINT8_C(    14),  UINT8_C(    38),  UINT8_C(   180),
         UINT8_C(    44),  UINT8_C(     4),  UINT8_C(   143),  UINT8_C(   172),
         UINT8_C(    51),  UINT8_C(    77),  UINT8_C(   188),  UINT8_C(    81) },
      {  INT8_C(     2), -INT8_C(     3),  INT8_C(     1),  INT8_C(     7),
         INT8_C(     1), -INT8_C(     7), -INT8_C(     3), -INT8_C(     3),
        -INT8_C(     6), -INT8_C(     6),  INT8_C(     0), -INT8_C(     5),
        -INT8_C(     7),  INT8_C(     0),  INT8_C(     1), -INT8_C(     7) },
      {  UINT8_C(   192),  UINT8_C(    14),  UINT8_C(   255),  UINT8_C(   255),
         UINT8_C(   255),  UINT8_C(     0),  UINT8_C(     5),  UINT8_C(    23),
         UINT8_C(     1),  UINT8_C(     0),  UINT8_C(   143),  UINT8_C(     5),
         UINT8_C(     0),  UINT8_C(    77),  UINT8_C(   255),  UINT8_C(     1) } },
    { {  UINT8_C(   145),  UINT8_C(   198),  UINT8_C(    59),  UINT8_C(   229),
         UINT8_C(   193),  UINT8_C(    40),  UINT8_C(   202),  UINT8_C(    18),
         UINT8_C(   189),  UINT8_C(   226),  UINT8_C(    53),  UINT8_C(   190),
         UINT8_C(   171),  UINT8_C(   119),  UINT8_C(   184),  UINT8_C(   228) },
      { -INT8_C(     5),  INT8_C(     2),  INT8_C(     3), -INT8_C(     1),
        -INT8_C(     6),  INT8_C(     6),  INT8_C(     0),  INT8_C(     5),
         INT8_C(     5),  INT8_C(     1), -INT8_C(     3),  INT8_C(     1),
        -INT8_C(     5),  INT8_C(     7), -INT8_C(     4),  INT8_C(     4) },
      {  UINT8_C(     5),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   115),
         UINT8_C(     3),  UINT8_C(   255),  UINT8_C(   202),  UINT8_C(   255),
         UINT8_C(   255),  UINT8_C(   255),  UINT8_C(     7),  UINT8_C(   255),
         UINT8_C(     5),  UINT8_C(   255),  UINT8_C(    12),  UINT8_C(   255) } },
    { {  UINT8_C(   144),  UINT8_C(    70),  UINT8_C(    74),  UINT8_C(     4),
         UINT8_C(   109),  UINT8_C(   142),  UINT8_C(   116),  UINT8_C(   159),
         UINT8_C(   142),  UINT8_C(    25),  UINT8_C(   177),  UINT8_C(   177),
         UINT8_C(    76),  UINT8_C(   150),  UINT8_C(    86),  UINT8_C(   238) },
      {  INT8_C(     0), -INT8_C(     3),  INT8_C(     3), -INT8_C(     3),
         INT8_C(     5), -INT8_C(     4), -INT8_C(     3), -INT8_C(     2),
         INT8_C(     3), -INT8_C(     4), -INT8_C(     3),  INT8_C(     3),
         INT8_C(     1),  INT8_C(     5),  INT8_C(     7),  INT8_C(     5) },
      {  UINT8_C(   144),  UINT8_C(     9),  UINT8_C(   255),  UINT8_C(     1),
         UINT8_C(   255),  UINT8_C(     9),  UINT8_C(    15),  UINT8_C(    40),
         UINT8_C(   255),  UINT8_C(     2),  UINT8_C(    22),  UINT8_C(   255),
         UINT8_C(   152),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   255) } },
    { {  UINT8_C(   199),  UINT8_C(   104),  UINT8_C(    74),  UINT8_C(    67),
         UINT8_C(   128),  UINT8_C(   140),  UINT8_C(    56),  UINT8_C(   104),
         UINT8_C(    88),  UINT8_C(   100),  UINT8_C(   110),  UINT8_C(    42),
         UINT8_C(   123),  UINT8_C(   111),  UINT8_C(   252),  UINT8_C(   156) },
      { -INT8_C(     8),  INT8_C(     0), -INT8_C(     7), -INT8_C(     1),
        -INT8_C(     8),  INT8_C(     4), -INT8_C(     5),  INT8_C(     4),
        -INT8_C(     2),  INT8_C(     5),  INT8_C(     0), -INT8_C(     6),
         INT8_C(     5), -INT8_C(     8),  INT8_C(     8),  INT8_C(     5) },
      {  UINT8_C(     1),  UINT8_C(   104),  UINT8_C(     1),  UINT8_C(    34),
         UINT8_C(     1),  UINT8_C(   255),  UINT8_C(     2),  UINT8_C(   255),
         UINT8_C(    22),  UINT8_C(   255),  UINT8_C(   110),  UINT8_C(     1),
         UINT8_C(   255),  UINT8_C(     0),  UINT8_C(   255),  UINT8_C(   255) } },
    { {  UINT8_C(   140),  UINT8_C(    24),  UINT8_C(    20),  UINT8_C(   116),
         UINT8_C(     5),  UINT8_C(   138),  UINT8_C(    42),  UINT8_C(   115),
         UINT8_C(   111),  UINT8_C(   129),  UINT8_C(   142),  UINT8_C(   213),
         UINT8_C(   155),  UINT8_C(   130),  UINT8_C(   150),  UINT8_C(   231) },
      {  INT8_C(     2), -INT8_C(     8), -INT8_C(     8), -INT8_C(     6),
         INT8_C(     1), -INT8_C(     5),  INT8_C(     1),  INT8_C(     2),
        -INT8_C(     1), -INT8_C(     2),  INT8_C(     5),  INT8_C(     5),
        -INT8_C(     3),  INT8_C(     6), -INT8_C(     1), -INT8_C(     6) },
      {  UINT8_C(   255),  UINT8_C(     0),  UINT8_C(     0),  UINT8_C(     2),
         UINT8_C(    10),  UINT8_C(     4),  UINT8_C(    84),  UINT8_C(   255),
         UINT8_C(    56),  UINT8_C(    32),  UINT8_C(   255),  UINT8_C(   255),
         UINT8_C(    19),  UINT8_C(   255),  UINT8_C(    75),  UINT8_C(     4) } },
    { {  UINT8_C(   123),  UINT8_C(    36),  UINT8_C(    41),  UINT8_C(   105),
         UINT8_C(    46),  UINT8_C(    64),  UINT8_C(   195),  UINT8_C(   242),
         UINT8_C(   155),  UINT8_C(   176),  UINT8_C(   255),  UINT8_C(   148),
         UINT8_C(    51),  UINT8_C(    35),  UINT8_C(   168),  UINT8_C(   226) },
      {  INT8_C(     0),  INT8_C(     6),  INT8_C(     5), -INT8_C(     6),
         INT8_C(     1), -INT8_C(     5), -INT8_C(     4),  INT8_C(     2),
         INT8_C(     6),  INT8_C(     3), -INT8_C(     1),  INT8_C(     7),
        -INT8_C(     2), -INT8_C(     2), -INT8_C(     1),  INT8_C(     4) },
      {  UINT8_C(   123),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(     2),
         UINT8_C(    92),  UINT8_C(     2),  UINT8_C(    12),  UINT8_C(   255),
         UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   128),  UINT8_C(   255),
         UINT8_C(    13),  UINT8_C(     9),  UINT8_C(    84),  UINT8_C(   255) } },
    { {  UINT8_C(   209),  UINT8_C(   246),  UINT8_C(   234),  UINT8_C(   223),
         UINT8_C(   203),  UINT8_C(   186),  UINT8_C(   139),  UINT8_C(   148),
         UINT8_C(   180),  UINT8_C(    18),  UINT8_C(     7),  UINT8_C(   172),
         UINT8_C(   225),  UINT8_C(    26),  UINT8_C(   149),  UINT8_C(    28) },
      {  INT8_C(     5), -INT8_C(     1),  INT8_C(     4),  INT8_C(     1),
         INT8_C(     1),  INT8_C(     1),  INT8_C(     7), -INT8_C(     5),
        -INT8_C(     2),  INT8_C(     4), -INT8_C(     7),  INT8_C(     1),
        -INT8_C(     3), -INT8_C(     2),  INT8_C(     7),  INT8_C(     0) },
      {  UINT8_C(   255),  UINT8_C(   123),  UINT8_C(   255),  UINT8_C(   255),
         UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   255),  UINT8_C(     5),
         UINT8_C(    45),  UINT8_C(   255),  UINT8_C(     0),  UINT8_C(   255),
         UINT8_C(    28),  UINT8_C(     7),  UINT8_C(   255),  UINT8_C(    28) } },
    { {  UINT8_C(   186),  UINT8_C(   147),  UINT8_C(    83),  UINT8_C(   114),
         UINT8_C(   130),  UINT8_C(    80),  UINT8_C(   252),  UINT8_C(    63),
         UINT8_C(   137),  UINT8_C(   214),  UINT8_C(   119),  UINT8_C(   227),
         UINT8_C(    29),  UINT8_C(   181),  UINT8_C(   148),  UINT8_C(    21) },
      {  INT8_C(     2),  INT8_C(     0),  INT8_C(     4),  INT8_C(     1),
         INT8_C(     8), -INT8_C(     6),  INT8_C(     3),  INT8_C(     6),
         INT8_C(     5), -INT8_C(     2), -INT8_C(     6),  INT8_C(     0),
         INT8_C(     6),  INT8_C(     5),  INT8_C(     0),  INT8_C(     7) },
      {  UINT8_C(   255),  UINT8_C(   147),  UINT8_C(   255),  UINT8_C(   228),
         UINT8_C(   255),  UINT8_C(     1),  UINT8_C(   255),  UINT8_C(   255),
         UINT8_C(   255),  UINT8_C(    54),  UINT8_C(     2),  UINT8_C(   227),
         UINT8_C(   255),  UINT8_C(   255),  UINT8_C(   148),  UINT8_C(   255) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint8x16_t a = simde_vld1q_u8(test_vec[i].a);
    simde_int8x16_t b = simde_vld1q_s8(test_vec[i].b);
    simde_uint8x16_t r = simde_vqrshlq_u8(a, b);

    simde_test_arm_neon_assert_equal_u8x16(r, simde_vld1q_u8(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint8x16_t a = simde_test_arm_neon_random_u8x16();
    simde_int8x16_t b = simde_test_arm_neon_random_i8x16();
    simde_uint8x16_t r = simde_vqrshlq_u8(a, b);

    simde_test_arm_neon_write_u8x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i8x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u8x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_u16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint16_t a[8];
    int16_t b[8];
    uint16_t r[8];
  } test_vec[] = {
    { {  UINT16_C(   61506),  UINT16_C(   19083),  UINT16_C(   60908),  UINT16_C(   40546),
         UINT16_C(   29927),  UINT16_C(   62112),  UINT16_C(   60795),  UINT16_C(   39381) },
      {  INT16_C(       5), -INT16_C(       9), -INT16_C(       2), -INT16_C(       8),
        -INT16_C(       7), -INT16_C(       4),  INT16_C(       4), -INT16_C(       9) },
      {  UINT16_C(   65535),  UINT16_C(      37),  UINT16_C(   15227),  UINT16_C(     158),
         UINT16_C(     234),  UINT16_C(    3882),  UINT16_C(   65535),  UINT16_C(      77) } },
    { {  UINT16_C(   52703),  UINT16_C(   30753),  UINT16_C(   21922),  UINT16_C(   13244),
         UINT16_C(   23660),  UINT16_C(   33835),  UINT16_C(    6910),  UINT16_C(   21793) },
      { -INT16_C(      11), -INT16_C(      12),  INT16_C(      15),  INT16_C(      11),
         INT16_C(       6), -INT16_C(       1), -INT16_C(      16), -INT16_C(       1) },
      {  UINT16_C(      26),  UINT16_C(       8),  UINT16_C(   65535),  UINT16_C(   65535),
         UINT16_C(   65535),  UINT16_C(   16918),  UINT16_C(       0),  UINT16_C(   10897) } },
    { {  UINT16_C(   49424),  UINT16_C(    1272),  UINT16_C(   11256),  UINT16_C(   15451),
         UINT16_C(   38439),  UINT16_C(   22519),  UINT16_C(   53624),  UINT16_C(   56406) },
      { -INT16_C(      11), -INT16_C(       3),  INT16_C(      10), -INT16_C(      10),
         INT16_C(      12),  INT16_C(       3),  INT16_C(       9),  INT16_C(       6) },
      {  UINT16_C(      24),  UINT16_C(     159),  UINT16_C(   65535),  UINT16_C(      15),
         UINT16_C(   65535),  UINT16_C(   65535),  UINT16_C(   65535),  UINT16_C(   65535) } },
    { {  UINT16_C(   34648),  UINT16_C(   38658),  UINT16_C(   18124),  UINT16_C(   29610),
         UINT16_C(   34257),  UINT16_C(   23288),  UINT16_C(    2479),  UINT16_C(   51223) },
      { -INT16_C(      11), -INT16_C(       3),  INT16_C(       4), -INT16_C(      10),
         INT16_C(       7), -INT16_C(       7),  INT16_C(      10),  INT16_C(       5) },
      {  UINT16_C(      17),  UINT16_C(    4832),  UINT16_C(   65535),  UINT16_C(      29),
         UINT16_C(   65535),  UINT16_C(     182),  UINT16_C(   65535),  UINT16_C(   65535) } },
    { {  UINT16_C(   34910),  UINT16_C(   29561),  UINT16_C(   57208),  UINT16_C(     321),
         UINT16_C(   59682),  UINT16_C(   10906),  UINT16_C(   56329),  UINT16_C(     122) },
      { -INT16_C(       9),  INT16_C(       4), -INT16_C(      16), -INT16_C(       1),
         INT16_C(      10), -INT16_C(       7), -INT16_C(       1), -INT16_C(       1) },
      {  UINT16_C(      68),  UINT16_C(   65535),  UINT16_C(       1),  UINT16_C(     161),
         UINT16_C(   65535),  UINT16_C(      85),  UINT16_C(   28165),  UINT16_C(      61) } },
    { {  UINT16_C(   17592),  UINT16_C(     628),  UINT16_C(   30585),  UINT16_C(   18746),
         UINT16_C(    2909),  UINT16_C(    4209),  UINT16_C(    6780),  UINT16_C(   29795) },
      { -INT16_C(      12),  INT16_C(       9), -INT16_C(      11), -INT16_C(       9),
         INT16_C(       8),  INT16_C(       0),  INT16_C(       1),  INT16_C(       7) },
      {  UINT16_C(       4),  UINT16_C(   65535),  UINT16_C(      15),  UINT16_C(      37),
         UINT16_C(   65535),  UINT16_C(    4209),  UINT16_C(   13560),  UINT16_C(   65535) } },
    { {  UINT16_C(   27761),  UINT16_C(   46906),  UINT16_C(   25315),  UINT16_C(   54517),
         UINT16_C(    3930),  UINT16_C(   38934),  UINT16_C(   33158),  UINT16_C(   15978) },
      { -INT16_C(       5),  INT16_C(       3), -INT16_C(      14), -INT16_C(       8),
        -INT16_C(      13),  INT16_C(      16), -INT16_C(       1), -INT16_C(      15) },
      {  UINT16_C(     868),  UINT16_C(   65535),  UINT16_C(       2),  UINT16_C(     213),
         UINT16_C(       0),  UINT16_C(   65535),  UINT16_C(   16579),  UINT16_C(       0) } },
    { {  UINT16_C(   52730),  UINT16_C(   21619),  UINT16_C(   31344),  UINT16_C(   36500),
         UINT16_C(   25393),  UINT16_C(   44945),  UINT16_C(   18048),  UINT16_C(   31672) },
      { -INT16_C(       6), -INT16_C(       9), -INT16_C(       7),  INT16_C(       1),
         INT16_C(       8), -INT16_C(       5),  INT16_C(       0),  INT16_C(       7) },
      {  UINT16_C(     824),  UINT16_C(      42),  UINT16_C(     245),  UINT16_C(   65535),
         UINT16_C(   65535),  UINT16_C(    1405),  UINT16_C(   18048),  UINT16_C(   65535) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint16x8_t a = simde_vld1q_u16(test_vec[i].a);
    simde_int16x8_t b = simde_vld1q_s16(test_vec[i].b);
    simde_uint16x8_t r = simde_vqrshlq_u16(a, b);
    simde_test_arm_neon_assert_equal_u16x8(r, simde_vld1q_u16(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint16x8_t a = simde_test_arm_neon_random_u16x8();
    simde_int16x8_t b = simde_test_arm_neon_random_i16x8();
    simde_uint16x8_t r = simde_vqrshlq_u16(a, b);

    simde_test_arm_neon_write_u16x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u16x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_u32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint32_t a[4];
    int32_t b[4];
    uint32_t r[4];
  } test_vec[] = {
    { {  UINT32_C(  1500975547),  UINT32_C(  2108068888),  UINT32_C(  2683398431),  UINT32_C(   367549241) },
      {  INT32_C(          32),  INT32_C(           4),  INT32_C(          12),  INT32_C(           7) },
      {  UINT32_C(  4294967295),  UINT32_C(  4294967295),  UINT32_C(  4294967295),  UINT32_C(  4294967295) } },
    { {  UINT32_C(  3541445283),  UINT32_C(  2942869736),  UINT32_C(  2791986498),  UINT32_C(  3847214939) },
      { -INT32_C(          15), -INT32_C(           2),  INT32_C(           3), -INT32_C(           5) },
      {  UINT32_C(      108076),  UINT32_C(   735717434),  UINT32_C(  4294967295),  UINT32_C(   120225467) } },
    { {  UINT32_C(  2758985098),  UINT32_C(  1301200120),  UINT32_C(  2500937308),  UINT32_C(  1482943544) },
      { -INT32_C(          15),  INT32_C(          26),  INT32_C(           4), -INT32_C(          31) },
      {  UINT32_C(       84198),  UINT32_C(  4294967295),  UINT32_C(  4294967295),  UINT32_C(           1) } },
    { {  UINT32_C(  1695104610),  UINT32_C(  2154042730),  UINT32_C(  1247813531),  UINT32_C(   539880566) },
      { -INT32_C(           5), -INT32_C(          21),  INT32_C(          12),  INT32_C(           6) },
      {  UINT32_C(    52972019),  UINT32_C(        1027),  UINT32_C(  4294967295),  UINT32_C(  4294967295) } },
    { {  UINT32_C(  4087492662),  UINT32_C(   442729018),  UINT32_C(   640768667),  UINT32_C(  1635123388) },
      {  INT32_C(           7), -INT32_C(           3),  INT32_C(           7),  INT32_C(          31) },
      {  UINT32_C(  4294967295),  UINT32_C(    55341127),  UINT32_C(  4294967295),  UINT32_C(  4294967295) } },
    { {  UINT32_C(  4260977946),  UINT32_C(  2996827211),  UINT32_C(   864415427),  UINT32_C(  4108108071) },
      { -INT32_C(          16),  INT32_C(           7),  INT32_C(          10), -INT32_C(          27) },
      {  UINT32_C(       65017),  UINT32_C(  4294967295),  UINT32_C(  4294967295),  UINT32_C(          31) } },
    { {  UINT32_C(  4249729102),  UINT32_C(  3434311192),  UINT32_C(  2916969933),  UINT32_C(  1847713867) },
      { -INT32_C(          31), -INT32_C(          15), -INT32_C(          21), -INT32_C(          21) },
      {  UINT32_C(           2),  UINT32_C(      104807),  UINT32_C(        1391),  UINT32_C(         881) } },
    { {  UINT32_C(  2544619547),  UINT32_C(  3247194592),  UINT32_C(  2391939879),  UINT32_C(  2988854179) },
      {  INT32_C(           5),  INT32_C(           5),  INT32_C(          30), -INT32_C(          29) },
      {  UINT32_C(  4294967295),  UINT32_C(  4294967295),  UINT32_C(  4294967295),  UINT32_C(           6) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint32x4_t a = simde_vld1q_u32(test_vec[i].a);
    simde_int32x4_t b = simde_vld1q_s32(test_vec[i].b);
    simde_uint32x4_t r = simde_vqrshlq_u32(a, b);

    simde_test_arm_neon_assert_equal_u32x4(r, simde_vld1q_u32(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint32x4_t a = simde_test_arm_neon_random_u32x4();
    simde_int32x4_t b = simde_test_arm_neon_random_i32x4();
    simde_uint32x4_t r = simde_vqrshlq_u32(a, b);

    simde_test_arm_neon_write_u32x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vqrshlq_u64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    uint64_t a[2];
    int64_t b[2];
    uint64_t r[2];
  } test_vec[] = {
    { {  UINT64_C(12257908650438209556),  UINT64_C(  734662789645664798) },
      { -INT64_C(                  53), -INT64_C(                  24) },
      {  UINT64_C(                1361),  UINT64_C(         43789314607) } },
    { {  UINT64_C(15401328157929644581),  UINT64_C(11927978306316211307) },
      {  INT64_C(                  54), -INT64_C(                  26) },
      {  UINT64_C(18446744073709551615),  UINT64_C(        177740727459) } },
    { {  UINT64_C( 9806006901024356755),  UINT64_C( 2220988353704575177) },
      {  INT64_C(                  18),  INT64_C(                  48) },
      {  UINT64_C(18446744073709551615),  UINT64_C(18446744073709551615) } },
    { {  UINT64_C(17911317278255258189),  UINT64_C(16943905161406011814) },
      {  INT64_C(                  60),  INT64_C(                   1) },
      {  UINT64_C(18446744073709551615),  UINT64_C(18446744073709551615) } },
    { {  UINT64_C( 2887718461795313073),  UINT64_C(17639433086009224754) },
      {  INT64_C(                  32),  INT64_C(                  23) },
      {  UINT64_C(18446744073709551615),  UINT64_C(18446744073709551615) } },
    { {  UINT64_C( 5907043747386205374),  UINT64_C(11542381187061170644) },
      {  INT64_C(                  57), -INT64_C(                  15) },
      {  UINT64_C(18446744073709551615),  UINT64_C(     352245519624670) } },
    { {  UINT64_C(15325393461537196969),  UINT64_C(16407202451519969415) },
      { -INT64_C(                  15), -INT64_C(                  50) },
      {  UINT64_C(     467693892258826),  UINT64_C(               14573) } },
    { {  UINT64_C(17890801295880430688),  UINT64_C(12355916188791687384) },
      { -INT64_C(                  38), -INT64_C(                  25) },
      {  UINT64_C(            65086356),  UINT64_C(        368234997654) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint64x2_t a = simde_vld1q_u64(test_vec[i].a);
    simde_int64x2_t b = simde_vld1q_s64(test_vec[i].b);
    simde_uint64x2_t r = simde_vqrshlq_u64(a, b);
    simde_test_arm_neon_assert_equal_u64x2(r, simde_vld1q_u64(test_vec[i].r));
  }

  return 0;

#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_uint64x2_t a = simde_test_arm_neon_random_u64x2();
    simde_int64x2_t b = simde_test_arm_neon_random_i64x2();
    simde_uint64x2_t r = simde_vqrshlq_u64(a, b);

    simde_test_arm_neon_write_u64x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_u64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}


SIMDE_TEST_FUNC_LIST_BEGIN
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlb_s8)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlh_s16)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshls_s32)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshld_s64)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlb_u8)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlh_u16)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshls_u32)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshld_u64)

SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_s8)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_s16)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_s32)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_s64)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_u8)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_u16)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_u32)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshl_u64)

SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_s8)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_s16)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_s32)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_s64)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_u8)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_u16)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_u32)
SIMDE_TEST_FUNC_LIST_ENTRY(vqrshlq_u64)
SIMDE_TEST_FUNC_LIST_END

#include "test-neon-footer.h"
