# fmt: off
from ase.atoms import Atoms
from ase.calculators.singlepoint import SinglePointCalculator
from ase.io.res import Res, read_res, write_res

testdata = """
TITL 23221-ZDsSsJoEW14-3 -2.7839600000000004 1 -31005.480500000001 \
0 0 254 (P1) n - 1
CELL 1.0 33.998511000000001 19.938282999999998 6.7999999999999998 \
90 90 89.981431000000001
LATT -1
SFAC C H
H 2 0.419886000 0.142864000 0.243969000 1.0
H 2 0.449139000 0.135683000 0.670836000 1.0
H 2 0.284474000 0.165357000 0.236375000 1.0
H 2 0.326693000 0.119478000 0.609186000 1.0
H 2 0.09748500 0.08739500 0.484880000 1.0
H 2 0.02442900 0.08613700 0.524346000 1.0
H 2 0.198055000 0.153611000 0.757328000 1.0
H 2 0.175429000 0.133959000 0.325265000 1.0
H 2 0.398735000 0.09679800 0.440684000 1.0
H 2 0.471401000 0.09914900 0.463208000 1.0
H 2 0.266722000 0.108997000 0.413839000 1.0
H 2 0.339879000 0.104829000 0.360835000 1.0
H 2 0.07338100 0.131550000 0.294675000 1.0
H 2 0.04794100 0.124129000 0.727225000 1.0
H 2 0.225211000 0.105804000 0.585777000 1.0
H 2 0.152409000 0.09545300 0.529334000 1.0
H 2 0.483562000 0.846988000 0.185931000 1.0
H 2 0.412203000 0.881051000 0.503357000 1.0
H 2 0.334140000 0.872886000 0.548314000 1.0
H 2 0.04479200 0.837706000 0.669058000 1.0
H 2 0.123455000 0.859052000 0.614144000 1.0
H 2 0.179423000 0.845556000 0.356632000 1.0
H 2 0.482063000 0.887844000 0.415000000 1.0
H 2 0.406955000 0.863603000 0.254068000 1.0
H 2 0.336814000 0.868085000 0.292107000 1.0
H 2 0.06542800 0.847667000 0.435941000 1.0
H 2 0.129674000 0.805885000 0.820551000 1.0
H 2 0.198388000 0.859228000 0.590843000 1.0
H 2 0.261362000 0.875683000 0.445400000 1.0
H 2 0.261638000 0.837239000 0.213067000 1.0
H 2 0.922430000 0.133381000 0.326458000 1.0
H 2 0.945221000 0.143541000 0.761125000 1.0
H 2 0.796117000 0.143497000 0.270665000 1.0
H 2 0.823422000 0.134958000 0.699562000 1.0
H 2 0.597937000 0.105913000 0.422276000 1.0
H 2 0.525383000 0.103158000 0.471170000 1.0
H 2 0.718676000 0.09931000 0.508464000 1.0
H 2 0.645826000 0.104835000 0.550140000 1.0
H 2 0.897235000 0.09693900 0.527122000 1.0
H 2 0.969022000 0.09324000 0.583187000 1.0
H 2 0.773836000 0.09751500 0.463093000 1.0
H 2 0.845975000 0.09728100 0.493986000 1.0
H 2 0.573655000 0.154922000 0.243396000 1.0
H 2 0.548290000 0.142797000 0.673915000 1.0
H 2 0.696822000 0.135939000 0.298545000 1.0
H 2 0.670484000 0.154425000 0.726137000 1.0
H 2 0.993967000 0.805476000 0.209721000 1.0
H 2 0.925002000 0.847111000 0.584177000 1.0
H 2 0.848353000 0.844978000 0.660109000 1.0
H 2 0.548459000 0.874976000 0.574384000 1.0
H 2 0.629129000 0.889040000 0.458485000 1.0
H 2 0.701418000 0.868349000 0.357223000 1.0
H 2 0.990692000 0.856777000 0.420401000 1.0
H 2 0.916085000 0.849572000 0.331757000 1.0
H 2 0.847830000 0.863225000 0.409754000 1.0
H 2 0.559835000 0.876727000 0.325807000 1.0
H 2 0.626500000 0.856782000 0.697646000 1.0
H 2 0.704670000 0.873144000 0.612507000 1.0
H 2 0.776989000 0.865351000 0.609983000 1.0
H 2 0.777771000 0.862294000 0.354035000 1.0
C 1 0.01794100 0.194517000 0.539562000 1.0
C 1 0.03913900 0.253871000 0.514899000 1.0
C 1 0.02459800 0.319523000 0.514860000 1.0
C 1 0.08047700 0.254851000 0.513051000 1.0
C 1 0.09390100 0.321005000 0.525070000 1.0
C 1 0.135544000 0.326265000 0.537909000 1.0
C 1 0.160118000 0.266948000 0.537269000 1.0
C 1 0.143168000 0.202253000 0.508563000 1.0
C 1 0.102721000 0.196410000 0.495795000 1.0
C 1 0.08016000 0.132766000 0.454453000 1.0
C 1 0.04134100 0.130791000 0.568495000 1.0
C 1 0.168324000 0.140886000 0.483336000 1.0
C 1 0.205677000 0.149395000 0.599179000 1.0
C 1 0.225390000 0.213268000 0.541040000 1.0
C 1 0.202924000 0.273399000 0.543543000 1.0
C 1 0.222717000 0.336658000 0.541106000 1.0
C 1 0.265722000 0.338220000 0.544351000 1.0
C 1 0.286146000 0.275315000 0.510965000 1.0
C 1 0.264627000 0.214628000 0.489721000 1.0
C 1 0.284365000 0.154659000 0.396444000 1.0
C 1 0.326395000 0.141923000 0.459913000 1.0
C 1 0.349834000 0.206050000 0.468495000 1.0
C 1 0.329052000 0.267406000 0.508318000 1.0
C 1 0.351096000 0.323391000 0.565284000 1.0
C 1 0.393102000 0.321771000 0.553431000 1.0
C 1 0.413186000 0.264423000 0.490903000 1.0
C 1 0.391056000 0.204937000 0.456991000 1.0
C 1 0.414597000 0.143259000 0.405011000 1.0
C 1 0.454381000 0.143030000 0.511205000 1.0
C 1 0.477501000 0.207590000 0.483482000 1.0
C 1 0.455793000 0.267949000 0.481510000 1.0
C 1 0.476229000 0.330977000 0.490515000 1.0
C 1 0.517927000 0.334708000 0.476110000 1.0
C 1 0.538971000 0.273269000 0.465467000 1.0
C 1 0.581176000 0.275318000 0.464382000 1.0
C 1 0.602425000 0.213636000 0.460758000 1.0
C 1 0.580370000 0.151600000 0.402822000 1.0
C 1 0.541989000 0.148064000 0.514347000 1.0
C 1 0.518444000 0.210779000 0.482198000 1.0
C 1 0.641824000 0.212664000 0.508219000 1.0
C 1 0.663763000 0.150242000 0.566920000 1.0
C 1 0.702588000 0.143874000 0.457599000 1.0
C 1 0.726468000 0.206767000 0.487713000 1.0
C 1 0.705521000 0.268832000 0.510616000 1.0
C 1 0.726720000 0.329679000 0.513874000 1.0
C 1 0.768508000 0.326543000 0.516794000 1.0
C 1 0.789560000 0.265699000 0.509986000 1.0
C 1 0.767597000 0.205421000 0.483013000 1.0
C 1 0.790573000 0.143396000 0.431567000 1.0
C 1 0.829573000 0.141851000 0.540459000 1.0
C 1 0.852990000 0.205809000 0.515729000 1.0
C 1 0.832248000 0.267723000 0.513436000 1.0
C 1 0.853934000 0.330250000 0.513810000 1.0
C 1 0.896612000 0.329331000 0.515192000 1.0
C 1 0.916234000 0.266229000 0.514897000 1.0
C 1 0.893705000 0.205356000 0.508679000 1.0
C 1 0.915210000 0.140161000 0.484505000 1.0
C 1 0.952772000 0.140428000 0.602463000 1.0
C 1 0.977331000 0.199804000 0.549262000 1.0
C 1 0.959053000 0.263208000 0.521439000 1.0
C 1 0.982836000 0.323414000 0.515000000 1.0
C 1 0.663404000 0.273466000 0.508025000 1.0
C 1 0.643457000 0.336491000 0.492667000 1.0
C 1 0.601415000 0.337627000 0.481582000 1.0
C 1 0.09613400 0.479995000 0.539265000 1.0
C 1 0.135896000 0.453141000 0.542245000 1.0
C 1 0.154500000 0.388296000 0.543473000 1.0
C 1 0.196896000 0.393408000 0.543828000 1.0
C 1 0.205657000 0.462762000 0.539655000 1.0
C 1 0.241364000 0.498860000 0.542657000 1.0
C 1 0.279121000 0.468631000 0.589891000 1.0
C 1 0.289883000 0.397780000 0.590070000 1.0
C 1 0.330855000 0.382727000 0.627307000 1.0
C 1 0.355317000 0.431754000 0.706274000 1.0
C 1 0.397022000 0.430985000 0.689661000 1.0
C 1 0.415866000 0.379639000 0.597475000 1.0
C 1 0.455416000 0.390709000 0.539969000 1.0
C 1 0.473164000 0.456528000 0.542346000 1.0
C 1 0.447730000 0.515284000 0.594838000 1.0
C 1 0.412433000 0.496298000 0.686780000 1.0
C 1 0.379001000 0.538334000 0.705783000 1.0
C 1 0.344264000 0.498496000 0.714088000 1.0
C 1 0.308706000 0.518284000 0.633411000 1.0
C 1 0.307240000 0.586831000 0.574246000 1.0
C 1 0.271443000 0.613497000 0.508288000 1.0
C 1 0.238141000 0.569912000 0.515445000 1.0
C 1 0.201204000 0.603510000 0.513332000 1.0
C 1 0.166253000 0.565735000 0.538533000 1.0
C 1 0.168848000 0.497096000 0.539230000 1.0
C 1 0.128660000 0.594729000 0.557453000 1.0
C 1 0.09402600 0.552572000 0.548038000 1.0
C 1 0.05757600 0.588352000 0.532770000 1.0
C 1 0.02152400 0.551578000 0.509056000 1.0
C 1 0.02048700 0.478858000 0.515228000 1.0
C 1 0.05851300 0.440042000 0.525788000 1.0
C 1 0.05888400 0.367680000 0.521406000 1.0
C 1 0.378536000 0.600675000 0.625366000 1.0
C 1 0.449475000 0.586932000 0.546085000 1.0
C 1 0.484310000 0.621585000 0.485403000 1.0
C 1 0.521296000 0.588152000 0.483997000 1.0
C 1 0.535536000 0.520951000 0.491653000 1.0
C 1 0.578664000 0.522255000 0.487048000 1.0
C 1 0.589538000 0.590300000 0.484659000 1.0
C 1 0.625482000 0.624676000 0.487672000 1.0
C 1 0.660699000 0.588479000 0.485887000 1.0
C 1 0.669402000 0.519599000 0.487599000 1.0
C 1 0.644748000 0.461151000 0.488946000 1.0
C 1 0.665196000 0.397748000 0.495207000 1.0
C 1 0.707365000 0.393348000 0.507693000 1.0
C 1 0.730917000 0.453181000 0.505280000 1.0
C 1 0.771816000 0.449346000 0.512143000 1.0
C 1 0.789813000 0.386183000 0.517783000 1.0
C 1 0.831256000 0.390806000 0.515080000 1.0
C 1 0.839385000 0.461422000 0.510979000 1.0
C 1 0.803229000 0.498721000 0.506717000 1.0
C 1 0.803046000 0.569184000 0.494652000 1.0
C 1 0.840663000 0.603268000 0.491666000 1.0
C 1 0.876373000 0.565370000 0.494733000 1.0
C 1 0.875832000 0.494281000 0.506735000 1.0
C 1 0.911563000 0.458087000 0.511290000 1.0
C 1 0.920912000 0.388659000 0.515311000 1.0
C 1 0.963362000 0.385116000 0.514150000 1.0
C 1 0.981113000 0.450779000 0.511755000 1.0
C 1 0.947657000 0.493847000 0.507161000 1.0
C 1 0.948852000 0.562366000 0.489902000 1.0
C 1 0.986523000 0.592473000 0.483853000 1.0
C 1 0.913107000 0.599703000 0.485384000 1.0
C 1 0.767622000 0.609578000 0.487796000 1.0
C 1 0.729166000 0.580335000 0.486762000 1.0
C 1 0.712088000 0.515526000 0.490898000 1.0
C 1 0.602087000 0.462219000 0.487277000 1.0
C 1 0.580271000 0.400158000 0.484177000 1.0
C 1 0.537581000 0.398662000 0.485171000 1.0
C 1 0.514602000 0.459384000 0.502213000 1.0
C 1 0.990939000 0.803735000 0.373410000 1.0
C 1 0.953580000 0.770110000 0.426145000 1.0
C 1 0.919118000 0.813418000 0.455956000 1.0
C 1 0.881770000 0.777151000 0.483940000 1.0
C 1 0.847096000 0.819629000 0.512592000 1.0
C 1 0.809644000 0.783435000 0.492531000 1.0
C 1 0.775342000 0.828788000 0.485485000 1.0
C 1 0.737479000 0.793289000 0.486370000 1.0
C 1 0.701834000 0.835982000 0.490995000 1.0
C 1 0.663877000 0.800103000 0.505701000 1.0
C 1 0.627655000 0.840784000 0.539693000 1.0
C 1 0.591383000 0.803037000 0.486762000 1.0
C 1 0.554733000 0.840652000 0.448253000 1.0
C 1 0.518891000 0.800524000 0.409871000 1.0
C 1 0.482917000 0.836989000 0.348107000 1.0
C 1 0.447079000 0.798463000 0.402285000 1.0
C 1 0.447920000 0.730820000 0.446717000 1.0
C 1 0.412942000 0.696106000 0.498425000 1.0
C 1 0.375825000 0.731701000 0.486834000 1.0
C 1 0.374259000 0.798522000 0.439397000 1.0
C 1 0.410117000 0.837760000 0.399457000 1.0
C 1 0.336384000 0.835680000 0.425650000 1.0
C 1 0.300964000 0.792387000 0.422190000 1.0
C 1 0.263011000 0.825530000 0.374205000 1.0
C 1 0.228873000 0.783388000 0.434272000 1.0
C 1 0.191879000 0.817881000 0.485715000 1.0
C 1 0.160643000 0.773960000 0.564103000 1.0
C 1 0.126050000 0.805850000 0.657127000 1.0
C 1 0.09056700 0.767672000 0.597612000 1.0
C 1 0.05630600 0.807865000 0.541752000 1.0
C 1 0.02436500 0.766728000 0.461523000 1.0
C 1 0.02330200 0.697908000 0.482438000 1.0
C 1 0.05729800 0.659846000 0.535401000 1.0
C 1 0.09144300 0.699123000 0.576601000 1.0
C 1 0.127749000 0.664359000 0.575836000 1.0
C 1 0.162179000 0.704817000 0.546855000 1.0
C 1 0.198453000 0.673358000 0.499095000 1.0
C 1 0.231773000 0.714679000 0.455751000 1.0
C 1 0.269070000 0.682171000 0.461557000 1.0
C 1 0.303211000 0.724424000 0.460036000 1.0
C 1 0.339999000 0.694453000 0.509757000 1.0
C 1 0.341334000 0.628677000 0.575314000 1.0
C 1 0.414614000 0.628788000 0.560270000 1.0
C 1 0.483592000 0.692141000 0.449201000 1.0
C 1 0.518732000 0.732106000 0.436492000 1.0
C 1 0.555089000 0.698529000 0.464888000 1.0
C 1 0.555016000 0.629689000 0.477655000 1.0
C 1 0.591384000 0.734166000 0.481431000 1.0
C 1 0.626381000 0.695017000 0.492822000 1.0
C 1 0.662628000 0.731573000 0.493304000 1.0
C 1 0.698206000 0.693358000 0.485021000 1.0
C 1 0.735934000 0.724332000 0.485093000 1.0
C 1 0.770097000 0.680802000 0.484874000 1.0
C 1 0.807458000 0.714133000 0.488127000 1.0
C 1 0.842410000 0.673725000 0.489316000 1.0
C 1 0.879581000 0.708390000 0.482618000 1.0
C 1 0.914733000 0.669363000 0.474005000 1.0
C 1 0.952018000 0.701703000 0.451744000 1.0
C 1 0.987402000 0.662138000 0.459246000 1.0
C 1 0.696482000 0.624775000 0.484183000 1.0
END
"""


def test_res():
    filepath = 'test.res'

    with open(filepath, 'w') as fd:
        fd.write(testdata)

    res = Res.from_file(filepath)
    assert res.atoms.get_chemical_formula() == 'C194H60'

    atoms = read_res(filepath)
    assert res.atoms == atoms
    assert res.energy == atoms.get_potential_energy()

    spc = SinglePointCalculator(atoms, energy=res.energy)
    atoms.calc = spc

    write_res('test2.res', atoms)
    atoms2 = read_res(filepath)
    assert atoms2 == atoms

    write_res('test3.res', atoms, write_info=False, significant_figures=9)
    atoms3 = read_res('test3.res')
    assert atoms3 == atoms

    res_string = """TITL
    CELL 1.0 1.0 1.0 1.0 90.0 90.0 90.0
    LATT -1
    SFAC Si F
    Si 1 0.000000 0.000000 0.000000 1.0
    F 2 0.750000 0.500000 0.750000 1.0"""
    res = Res.from_string(res_string)
    assert res.atoms.get_chemical_formula() == 'FSi'
    assert len(res.atoms) == 2

    struct = Atoms(cell=[2.5, 3.5, 7.0],
                   symbols=['Na', 'Cl'],
                   positions=[[1.0, 0.0, 0.0], [0.0, 1.0, 0.0]])
    res = Res(struct)
    res_string = str(res)
    lines = res_string.splitlines()
    assert lines[1] == ('CELL 1.0 2.500000 3.500000 7.000000 '
                        '90.000000 90.000000 90.000000')
